<x-tenant-layout>
    <x-slot name="header">
        <h2 class="font-semibold text-xl text-gray-800 leading-tight">
            {{ __('Point of Sale') }}
        </h2>
    </x-slot>

    <div class="h-[calc(100vh-8rem)] flex overflow-hidden" x-data="posApp()">
        <!-- Product Catalog (Left) -->
        <div class="flex-1 flex flex-col bg-gray-50 border-r overflow-hidden">
            <!-- Search Bar -->
            <div class="p-4 bg-white shadow-sm z-10 flex gap-4">
                <input type="text" x-model="search" placeholder="Search products by name or SKU..." class="flex-1 rounded-md border-gray-300 shadow-sm focus:border-green-500 focus:ring focus:ring-green-200 focus:ring-opacity-50">
                
                <select x-model="categoryFilter" class="w-48 rounded-md border-gray-300 shadow-sm focus:border-green-500 focus:ring focus:ring-green-200 focus:ring-opacity-50">
                    <option value="">All Categories</option>
                    @foreach($categories as $category)
                        <option value="{{ $category->id }}">{{ $category->name }}</option>
                    @endforeach
                </select>
            </div>

            <!-- Product Grid -->
            <div class="flex-1 overflow-y-auto p-4">
                <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                    <template x-for="product in filteredProducts" :key="product.id">
                        <div @click="product.has_variants ? showVariantModal(product) : addSingleVariantToCart(product)" class="bg-white rounded-lg shadow cursor-pointer hover:shadow-md transition border border-gray-200 flex flex-col h-full relative group">
                            <!-- Variant Badge -->
                            <div x-show="product.has_variants" class="absolute top-2 left-2 px-2 py-1 bg-blue-600 text-white text-xs rounded opacity-75">
                                <span x-text="product.variants.length + ' variants'"></span>
                            </div>

                            <!-- Stock Badge -->
                            <div class="absolute top-2 right-2 px-2 py-1 text-white text-xs rounded opacity-75" 
                                 :class="getTotalStock(product) > 10 ? 'bg-green-600' : getTotalStock(product) > 0 ? 'bg-yellow-600' : 'bg-red-600'">
                                <span x-text="getTotalStock(product)"></span> in stock
                            </div>

                            <div class="h-32 bg-gray-100 rounded-t-lg flex items-center justify-center text-4xl overflow-hidden">
                                <template x-if="product.image">
                                    <img :src="product.image" class="w-full h-full object-cover">
                                </template>
                                <template x-if="!product.image">
                                    <span>📦</span>
                                </template>
                            </div>
                            <div class="p-3 flex-1 flex flex-col">
                                <h4 class="font-semibold text-gray-800 text-sm mb-1 leading-tight" x-text="product.name"></h4>
                                <div class="text-xs text-gray-500 mb-2">
                                    <span x-show="!product.has_variants" x-text="'SKU: ' + product.variants[0]?.sku"></span>
                                    <span x-show="product.has_variants" x-text="'Multiple variants available'"></span>
                                </div>
                                <div class="mt-auto flex justify-between items-center">
                                    <span class="font-bold text-green-600 text-lg" x-text="'$' + getMinPrice(product).toFixed(2)"></span>
                                    <span class="text-xs text-gray-400" x-text="product.variants[0]?.unit || ''"></span>
                                </div>
                            </div>
                        </div>
                    </template>
                </div>
            </div>
        </div>

        <!-- Cart (Right) -->
        <div class="w-[450px] flex flex-col bg-white shadow-xl z-20 border-l border-gray-200">
            <!-- Customer Selection -->
            <div class="p-4 border-b bg-gray-50">
                <div class="flex gap-2 mb-2">
                    <select x-model="customerType" class="w-1/3 rounded-md border-gray-300 text-sm shadow-sm focus:border-green-500 focus:ring-green-200">
                        <option value="all">All Types</option>
                        <option value="Customer">Customer</option>
                        <option value="Sub-Dealer">Sub-Dealer</option>
                    </select>
                    <select x-model="customerId" class="flex-1 rounded-md border-gray-300 text-sm shadow-sm focus:border-green-500 focus:ring-green-200">
                        <option value="">-- Guest Customer --</option>
                        <template x-for="customer in filteredCustomers" :key="customer.id">
                            <option :value="customer.id" x-text="customer.name + (customer.type === 'Sub-Dealer' ? ' (Sub)' : '')"></option>
                        </template>
                    </select>
                    <button @click="showAddCustomerModal = true" class="px-3 py-2 bg-blue-600 text-white rounded hover:bg-blue-700" title="Add Customer">+</button>
                </div>
            </div>

            <!-- Add Customer Modal -->
            <div x-show="showAddCustomerModal" class="fixed inset-0 z-50 overflow-y-auto" style="display: none;">
                <div class="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
                    <div class="fixed inset-0 transition-opacity" aria-hidden="true">
                        <div class="absolute inset-0 bg-gray-500 opacity-75"></div>
                    </div>
                    <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
                    <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                        <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                            <h3 class="text-lg leading-6 font-medium text-gray-900 mb-4">Add New Customer</h3>
                            <div class="space-y-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700">Name</label>
                                    <input type="text" x-model="newCustomer.name" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700">Phone</label>
                                    <input type="text" x-model="newCustomer.phone" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 sm:text-sm">
                                </div>
                                <div class="flex items-center">
                                    <input type="checkbox" x-model="newCustomer.is_sub_dealer" class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                                    <label class="ml-2 block text-sm text-gray-900">Is Sub-Dealer?</label>
                                </div>
                            </div>
                        </div>
                        <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                            <button @click="addNewCustomer" type="button" class="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-blue-600 text-base font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 sm:ml-3 sm:w-auto sm:text-sm">
                                Save Customer
                            </button>
                            <button @click="showAddCustomerModal = false" type="button" class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">
                                Cancel
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Variant Selection Modal -->
            <div x-show="showVariantModal" class="fixed inset-0 z-50 overflow-y-auto" style="display: none;">
                <div class="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
                    <div class="fixed inset-0 transition-opacity" aria-hidden="true">
                        <div class="absolute inset-0 bg-gray-500 opacity-75"></div>
                    </div>
                    <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
                    <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-2xl sm:w-full">
                        <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                            <h3 class="text-lg leading-6 font-medium text-gray-900 mb-4" x-text="'Select Variant - ' + (selectedProduct?.name || '')"></h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 max-h-96 overflow-y-auto">
                                <template x-for="variant in selectedProduct?.variants || []" :key="variant.id">
                                    <div @click="selectVariant(variant)" class="border rounded-lg p-4 cursor-pointer hover:border-green-500 hover:bg-green-50 transition">
                                        <div class="flex justify-between items-start mb-2">
                                            <h4 class="font-semibold text-gray-800" x-text="variant.name"></h4>
                                            <span class="px-2 py-1 text-white text-xs rounded" 
                                                  :class="variant.stock > 10 ? 'bg-green-600' : variant.stock > 0 ? 'bg-yellow-600' : 'bg-red-600'" 
                                                  x-text="variant.stock + ' in stock'"></span>
                                        </div>
                                        <div class="text-sm text-gray-500 mb-2" x-text="'SKU: ' + variant.sku"></div>
                                        <div class="text-lg font-bold text-green-600" x-text="'$' + Number(getVariantPrice(variant)).toFixed(2)"></div>
                                        <div class="text-xs text-gray-400" x-text="'Per ' + variant.unit"></div>
                                    </div>
                                </template>
                            </div>
                        </div>
                        <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                            <button @click="showVariantModal = false" type="button" class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">
                                Cancel
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Cart Items -->
            <div class="flex-1 overflow-y-auto p-4 space-y-3 bg-white">
                <template x-if="cart.length === 0">
                    <div class="text-center text-gray-400 mt-20">
                        <div class="text-6xl mb-4 opacity-50">🛒</div>
                        <p class="text-lg">Cart is empty</p>
                        <p class="text-sm">Scan barcode or click products</p>
                    </div>
                </template>
                <template x-for="(item, index) in cart" :key="item.id">
                    <div class="flex justify-between items-start border-b border-gray-100 pb-3 last:border-0">
                        <div class="flex-1 pr-2">
                            <h5 class="text-sm font-semibold text-gray-800" x-text="item.name"></h5>
                            
                            <!-- Unit Selector -->
                            <div class="mt-1 flex items-center space-x-2">
                                <select x-model="item.selected_unit_id" @change="updateUnit(item)" class="text-xs p-1 border rounded bg-gray-50 focus:ring-green-500 focus:border-green-500">
                                    <template x-for="unit in item.available_units" :key="unit.id">
                                        <option :value="unit.id" x-text="unit.name"></option>
                                    </template>
                                </select>
                            </div>

                            <div class="text-xs text-gray-500 mt-1 flex items-center">
                                <span x-text="'$' + Number(item.price).toFixed(2)"></span>
                                <span class="mx-1">x</span>
                                <input type="number" x-model.number="item.quantity" @change="updateCartTotal()" class="w-16 p-1 text-center border rounded text-xs mx-1">
                                <span class="mx-1">=</span>
                                <span class="font-bold text-gray-700" x-text="'$' + (item.price * item.quantity - (item.discount || 0)).toFixed(2)"></span>
                            </div>
                            
                            <!-- Item Discount -->
                            <div class="mt-1 flex items-center gap-1">
                                <span class="text-[10px] text-gray-400">Disc:</span>
                                <select x-model="item.discount_type" @change="calculateItemDiscount(item)" class="text-[10px] p-0.5 border rounded">
                                    <option value="none">None</option>
                                    <option value="fixed">Fixed</option>
                                    <option value="percent">%</option>
                                </select>
                                <input x-show="item.discount_type !== 'none'" type="number" x-model.number="item.discount_input" @input="calculateItemDiscount(item)" class="w-12 p-0.5 text-[10px] border rounded" placeholder="0">
                                <span x-show="item.discount > 0" class="text-[10px] text-red-500 ml-1" x-text="'-$' + item.discount.toFixed(2)"></span>
                            </div>
                        </div>
                        <button @click="removeFromCart(index)" class="text-red-400 hover:text-red-600 p-1 rounded hover:bg-red-50">
                            <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                            </svg>
                        </button>
                    </div>
                </template>
            </div>

            <!-- Totals & Checkout -->
            <div class="p-4 bg-gray-50 border-t shadow-[0_-4px_6px_-1px_rgba(0,0,0,0.1)]">
                <!-- Summary -->
                <div class="space-y-2 mb-4 text-sm">
                    <div class="flex justify-between text-gray-600">
                        <span>Subtotal</span>
                        <span class="font-medium" x-text="'$' + subtotal.toFixed(2)"></span>
                    </div>
                    <div class="flex justify-between text-gray-600 items-center">
                        <span>Discount</span>
                        <input type="number" x-model.number="discount" class="w-20 p-1 text-right border rounded text-sm bg-white" placeholder="0.00">
                    </div>
                    <div class="flex justify-between text-gray-600">
                        <span>Tax (0%)</span>
                        <span class="font-medium">$0.00</span>
                    </div>
                    <div class="flex justify-between text-xl font-bold text-gray-900 border-t pt-2 mt-2">
                        <span>Total Payable</span>
                        <span x-text="'$' + total.toFixed(2)"></span>
                    </div>
                </div>

                <!-- Payment Inputs -->
                <div class="mb-4 space-y-3">
                    <div class="grid grid-cols-3 gap-2">
                        <button @click="setPaymentType('cash')" :class="{'bg-green-600 text-white': paymentType === 'cash', 'bg-white border text-gray-700': paymentType !== 'cash'}" class="py-2 rounded font-medium text-sm transition">Cash</button>
                        <button @click="setPaymentType('card')" :class="{'bg-blue-600 text-white': paymentType === 'card', 'bg-white border text-gray-700': paymentType !== 'card'}" class="py-2 rounded font-medium text-sm transition">Card</button>
                        <button @click="setPaymentType('due')" :class="{'bg-orange-600 text-white': paymentType === 'due', 'bg-white border text-gray-700': paymentType !== 'due'}" class="py-2 rounded font-medium text-sm transition">Credit/Due</button>
                    </div>

                    <div x-show="paymentType === 'card'" class="flex items-center gap-2">
                         <label class="text-sm font-medium text-gray-700 w-20">Trans ID:</label>
                         <input type="text" x-model="transactionId" placeholder="Enter Transaction ID" class="flex-1 rounded border-gray-300 p-2 text-sm">
                    </div>

                    <div x-show="paymentType !== 'card'" class="flex items-center gap-2">
                        <label class="text-sm font-medium text-gray-700 w-20">Paid:</label>
                        <input type="number" x-model.number="paidAmount" class="flex-1 rounded border-gray-300 p-2 text-right font-bold text-lg" @input="calculateChange()">
                    </div>
                    
                    <div x-show="paymentType === 'cash' && changeAmount > 0" class="flex justify-between text-green-600 font-bold text-sm bg-green-50 p-2 rounded">
                        <span>Change Return:</span>
                        <span x-text="'$' + changeAmount.toFixed(2)"></span>
                    </div>

                    <div x-show="paymentType === 'due'" class="p-2 bg-orange-50 rounded text-sm text-orange-800">
                        <p><strong>Note:</strong> Partial payment is allowed. Remaining amount will be added to due.</p>
                    </div>
                </div>

                <!-- Actions -->
                <div class="grid grid-cols-2 gap-3">
                    <button 
                        @click="holdSale" 
                        class="py-3 bg-yellow-500 text-white rounded-lg font-bold hover:bg-yellow-600 transition flex justify-center items-center gap-2"
                        :disabled="cart.length === 0"
                    >
                        <span>⏸️</span> Hold
                    </button>
                    <button 
                        @click="checkout('completed')" 
                        :disabled="cart.length === 0 || loading || (paymentType !== 'due' && paidAmount < total)"
                        class="py-3 bg-green-600 text-white rounded-lg font-bold shadow hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed flex justify-center items-center gap-2"
                    >
                        <span x-show="!loading">✅ Pay & Print</span>
                        <span x-show="loading" class="animate-spin h-5 w-5 border-2 border-white border-t-transparent rounded-full"></span>
                    </button>
                </div>
            </div>
        </div>
    </div>

@section('scripts')
    <script>
        document.addEventListener('alpine:init', () => {
            Alpine.data('posApp', () => ({
                search: '',
                categoryFilter: '',
                products: @json($posProducts),
                customers: @json($customers),
                cart: [],
                customerId: '',
                customerType: 'all',
                
                // Add Customer State
                showAddCustomerModal: false,
                newCustomer: {
                    name: '',
                    phone: '',
                    is_sub_dealer: false
                },
                
                // Payment State
                subtotal: 0,
                discount: 0,
                total: 0,
                paymentType: 'cash',
                transactionId: '',
                paidAmount: 0,
                changeAmount: 0,
                
                loading: false,
                
                // Variant Selection State
                showVariantModal: false,
                selectedProduct: null,

                init() {
                    // Watch for cart changes to update totals
                    this.$watch('cart', () => this.updateCartTotal());
                    this.$watch('discount', () => this.updateCartTotal());
                },
                
                // Helper functions for product display
                getTotalStock(product) {
                    if (!product.variants) return 0;
                    return product.variants.reduce((total, variant) => total + (variant.stock || 0), 0);
                },
                
                getMinPrice(product) {
                    if (!product.variants || product.variants.length === 0) return 0;
                    const prices = [];
                    product.variants.forEach(variant => {
                        if (variant.prices && variant.prices.length > 0) {
                            // Get the base unit price or the first price
                            const basePrice = variant.prices.find(p => p.productUnit && p.productUnit.is_base);
                            if (basePrice) {
                                prices.push(basePrice.price);
                            } else if (variant.prices[0]) {
                                prices.push(variant.prices[0].price);
                            }
                        } else if (variant.price) {
                            // Fallback to legacy price field
                            prices.push(variant.price);
                        }
                    });
                    return prices.length > 0 ? Math.min(...prices) : 0;
                },
                
                getVariantPrice(variant) {
                    if (variant.prices && variant.prices.length > 0) {
                        // Get the base unit price or the first price
                        const basePrice = variant.prices.find(p => p.productUnit && p.productUnit.is_base);
                        if (basePrice) {
                            return basePrice.price;
                        } else if (variant.prices[0]) {
                            return variant.prices[0].price;
                        }
                    } else if (variant.price) {
                        // Fallback to legacy price field
                        return variant.price;
                    }
                    return 0;
                },
                
                buildAvailableUnits(variant) {
                    const units = [];
                    
                    // Add base unit
                    let basePrice = this.getVariantPrice(variant);
                    if (variant.prices && variant.prices.length > 0) {
                        const baseUnitPrice = variant.prices.find(p => p.productUnit && p.productUnit.is_base);
                        if (baseUnitPrice) {
                            basePrice = baseUnitPrice.price;
                        }
                    }
                    
                    units.push({
                        id: 'base',
                        name: variant.unit || 'Unit',
                        price: basePrice,
                        factor: 1,
                        is_base: true
                    });
                    
                    // Add other units from variant prices
                    if (variant.prices && variant.prices.length > 0) {
                        variant.prices.forEach(price => {
                            if (price.productUnit && !price.productUnit.is_base) {
                                units.push({
                                    id: price.productUnit.id.toString(),
                                    name: price.productUnit.name,
                                    price: price.price,
                                    factor: price.productUnit.conversion_factor || 1,
                                    is_base: false
                                });
                            }
                        });
                    }
                    
                    return units;
                },
                
                // Variant selection functions
                showVariantModal(product) {
                    this.selectedProduct = product;
                    this.showVariantModal = true;
                },
                
                selectVariant(variant) {
                    this.addVariantToCart(variant);
                    this.showVariantModal = false;
                    this.selectedProduct = null;
                },
                
                addSingleVariantToCart(product) {
                    if (product.variants && product.variants.length > 0) {
                        this.addVariantToCart(product.variants[0]);
                    }
                },
                
                addVariantToCart(variant) {
                    // Check if variant is in stock
                    if (variant.stock <= 0) {
                        alert('This variant is out of stock!');
                        return;
                    }
                    
                    const existingItem = this.cart.find(item => item.id === variant.id);
                    if (existingItem) {
                        // Check if adding more would exceed available stock
                        if (existingItem.quantity >= variant.stock) {
                            alert('Cannot add more. Only ' + variant.stock + ' in stock!');
                            return;
                        }
                        existingItem.quantity++;
                    } else {
                        // Create a clean copy of the variant for the cart
                        const newItem = {
                            id: variant.id,
                            product_id: variant.product_id,
                            name: variant.name,
                            sku: variant.sku,
                            barcode: variant.barcode,
                            price: parseFloat(this.getVariantPrice(variant)),
                            image: variant.image,
                            unit: variant.unit,
                            category_id: variant.category_id,
                            available_units: this.buildAvailableUnits(variant), // Build units with pricing
                            
                            // Cart specific fields
                            selected_unit_id: 'base',
                            conversion_factor: 1,
                            quantity: 1,
                            discount_type: 'none',
                            discount_input: 0,
                            discount: 0
                        };
                        
                        // Initialize with Base Unit
                        const baseUnit = newItem.available_units.find(u => u.id === 'base');
                        if (baseUnit) {
                            newItem.price = parseFloat(baseUnit.price);
                        }
                        
                        this.cart.push(newItem);
                    }
                    this.updateCartTotal();
                },
                
                setPaymentType(type) {
                    this.paymentType = type;
                    if (type === 'due') {
                        // Check if customer is selected
                        if (!this.customerId) {
                            alert('Please select a customer first for Credit/Due sales!');
                            this.paymentType = 'cash';
                            return;
                        }
                        this.paidAmount = 0; // Default to full credit
                    } else {
                        this.paidAmount = this.total;
                    }
                    this.calculateChange();
                },

                get filteredProducts() {
                    let filtered = this.products || [];
                    
                    if (this.categoryFilter !== '') {
                        filtered = filtered.filter(p => p.category_id == this.categoryFilter);
                    }

                    if (this.search !== '') {
                        const lowerSearch = this.search.toLowerCase();
                        filtered = filtered.filter(p => 
                            p.name.toLowerCase().includes(lowerSearch) || 
                            (p.sku && String(p.sku).toLowerCase().includes(lowerSearch)) ||
                            (p.barcode && String(p.barcode).toLowerCase().includes(lowerSearch))
                        );
                    }
                    
                    return filtered;
                },

                get filteredCustomers() {
                    if (this.customerType === 'all') return this.customers;
                    return this.customers.filter(c => c.type === this.customerType);
                },

                calculateItemDiscount(item) {
                    const itemTotal = item.price * item.quantity;
                    if (item.discount_type === 'fixed') {
                        item.discount = Math.min(itemTotal, item.discount_input);
                    } else if (item.discount_type === 'percent') {
                        item.discount = itemTotal * (item.discount_input / 100);
                    } else {
                        item.discount = 0;
                    }
                    this.updateCartTotal();
                },

                updateUnit(item) {
                    const unit = item.available_units.find(u => u.id == item.selected_unit_id);
                    if (unit) {
                        item.price = parseFloat(unit.price);
                        item.conversion_factor = parseFloat(unit.factor);
                    }
                    this.updateCartTotal();
                },

                removeFromCart(index) {
                    this.cart.splice(index, 1);
                    this.updateCartTotal();
                },

                updateCartTotal() {
                    this.subtotal = this.cart.reduce((sum, item) => sum + (item.price * item.quantity - (item.discount || 0)), 0);
                    this.total = Math.max(0, this.subtotal - this.discount);
                    
                    // Auto-update paid amount if paying full cash/card
                    if (this.paymentType !== 'due' && this.paidAmount === 0 && this.total > 0) {
                        this.paidAmount = this.total;
                    }
                    this.calculateChange();
                },

                calculateChange() {
                    if (this.paymentType === 'cash') {
                        this.changeAmount = Math.max(0, this.paidAmount - this.total);
                    } else {
                        this.changeAmount = 0;
                    }
                },

                holdSale() {
                    // Logic to save to local storage or hold list
                    alert('Sale held successfully! (Feature in progress)');
                    this.cart = [];
                    this.updateCartTotal();
                },

                checkout(status) {
                    if (this.cart.length === 0) return;
                    
                    // Validation
                    if (this.paymentType === 'due' && !this.customerId) {
                        alert('Please select a customer for Credit/Due sales.');
                        return;
                    }
                    if (this.paymentType === 'card' && !this.transactionId) {
                        alert('Please enter Transaction ID for card payments.');
                        return;
                    }

                    this.loading = true;
                    
                    const payload = {
                        customer_id: this.customerId,
                        items: this.cart.map(item => ({
                            variant_id: item.id,
                            quantity: item.quantity,
                            price: item.price,
                            unit_id: item.selected_unit_id === 'base' ? null : item.selected_unit_id,
                            discount_type: item.discount_type,
                            discount_value: item.discount_input
                        })),
                        subtotal: this.subtotal,
                        discount: this.discount,
                        total: this.total,
                        payment_type: this.paymentType,
                        transaction_id: this.transactionId,
                        paid_amount: this.paymentType === 'due' ? 0 : this.paidAmount,
                        note: '', // Add note input if needed
                        status: status
                    };

                    $.ajax({
                        url: '{{ route("tenant.pos.store") }}',
                        method: 'POST',
                        contentType: 'application/json',
                        headers: {
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                        },
                        data: JSON.stringify(payload),
                        success: (result) => {
                            if (result.success) {
                                alert('Sale completed! Printing receipt... (Mock)');
                                this.resetPos();
                            } else {
                                alert('Error: ' + result.message);
                            }
                        },
                        error: (xhr, status, error) => {
                            console.error('Checkout error:', error);
                            alert('An unexpected error occurred: ' + (xhr.responseJSON?.message || error));
                        },
                        complete: () => {
                            this.loading = false;
                        }
                    });
                },

                resetPos() {
                    this.cart = [];
                    this.customerId = '';
                    this.discount = 0;
                    this.paidAmount = 0;
                    this.paymentType = 'cash';
                    this.updateCartTotal();
                },

                addNewCustomer() {
                    if (!this.newCustomer.name) {
                        alert('Name is required');
                        return;
                    }

                    $.ajax({
                        url: '{{ route("tenant.customers.store") }}', // Assuming you have a resource route or create a specific one
                        method: 'POST',
                        contentType: 'application/json',
                        headers: {
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                        },
                        data: JSON.stringify(this.newCustomer),
                        success: (result) => {
                            if (result.success || result.id) { // Adapt based on your CustomerController response
                                const customer = result.customer || result; // Adapt based on response
                                
                                // Add to local list
                                const newC = {
                                    id: customer.id,
                                    name: customer.name,
                                    type: customer.is_sub_dealer ? 'Sub-Dealer' : 'Customer'
                                };
                                this.customers.push(newC);
                                this.customers.sort((a, b) => a.name.localeCompare(b.name));
                                
                                // Select new customer
                                this.customerId = newC.id;
                                
                                // Reset and Close
                                this.newCustomer = { name: '', phone: '', is_sub_dealer: false };
                                this.showAddCustomerModal = false;
                                alert('Customer added successfully!');
                            } else {
                                alert('Failed to add customer');
                            }
                        },
                        error: (xhr) => {
                            console.error('Add customer error:', xhr);
                            alert('Error adding customer: ' + (xhr.responseJSON?.message || 'Unknown error'));
                        }
                    });
                }
            }));
        });
    </script>
@endsection
</x-tenant-layout>
