@extends('layouts.developer')

@section('title', 'System Logs')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
    <!-- Page Header -->
    <div class="mb-8 flex flex-col md:flex-row md:items-end md:justify-between gap-4">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">System Logs</h1>
            <p class="mt-2 text-gray-600">
                @if($viewAll)
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-indigo-100 text-indigo-800 mr-2">Full View</span>
                    Showing all parsed logs from the last 5MB of log file.
                @else
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 mr-2">Recent View</span>
                    Showing the last 100 log entries.
                @endif
            </p>
        </div>
        <div class="flex flex-wrap gap-3">
            @if(!$viewAll)
                <a href="{{ route('developer.logs', ['view' => 'all']) }}" class="inline-flex items-center px-4 py-2 bg-indigo-600 text-white text-sm font-medium rounded-md hover:bg-indigo-700 transition">
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                    </svg>
                    View All Logs
                </a>
            @else
                <a href="{{ route('developer.logs') }}" class="inline-flex items-center px-4 py-2 bg-gray-600 text-white text-sm font-medium rounded-md hover:bg-gray-700 transition">
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 15l-3-3m0 0l3-3m-3 3h8M3 12a9 9 0 1118 0 8.959 8.959 0 01-18 0z" />
                    </svg>
                    Back to Recent
                </a>
            @endif
            <a href="{{ route('developer.logs.download') }}" class="inline-flex items-center px-4 py-2 bg-green-600 text-white text-sm font-medium rounded-md hover:bg-green-700 transition duration-150 ease-in-out">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
                </svg>
                Download
            </a>
            <form action="{{ route('developer.logs.clear') }}" method="POST" onsubmit="return confirm('Are you sure you want to clear all application logs? This action cannot be undone.');" class="inline">
                @csrf
                <button type="submit" class="inline-flex items-center px-4 py-2 bg-red-600 text-white text-sm font-medium rounded-md hover:bg-red-700 transition duration-150 ease-in-out">
                    <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                    </svg>
                    Clear Logs
                </button>
            </form>
        </div>
    </div>

    @if($viewAll)
    <!-- Log Categorization Summary -->
    <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-8 gap-4 mb-6">
        @php
            $levels = [
                'emergency' => ['bg' => 'bg-red-600', 'text' => 'text-white'],
                'alert' => ['bg' => 'bg-red-500', 'text' => 'text-white'],
                'critical' => ['bg' => 'bg-red-400', 'text' => 'text-white'],
                'error' => ['bg' => 'bg-red-100', 'text' => 'text-red-800'],
                'warning' => ['bg' => 'bg-yellow-100', 'text' => 'text-yellow-800'],
                'notice' => ['bg' => 'bg-purple-100', 'text' => 'text-purple-800'],
                'info' => ['bg' => 'bg-blue-100', 'text' => 'text-blue-800'],
                'debug' => ['bg' => 'bg-gray-100', 'text' => 'text-gray-800'],
            ];
        @endphp
        @foreach($levels as $level => $colors)
            <div class="bg-white rounded-lg shadow-sm border border-gray-100 p-3 flex flex-col items-center cursor-pointer hover:shadow-md transition" onclick="document.getElementById('logLevel').value = '{{ $level }}'; filterLogs();">
                <span class="text-xs font-bold uppercase text-gray-500 mb-1">{{ $level }}</span>
                <span class="text-lg font-bold {{ $colors['text'] }} {{ $colors['bg'] }} px-2 rounded-full min-w-[2rem] text-center">
                    {{ $logs->where('level', $level)->count() }}
                </span>
            </div>
        @endforeach
    </div>

    <!-- Log Filters -->
    <div class="bg-white shadow rounded-lg mb-6">
        <div class="px-4 py-5 sm:px-6">
            <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-3 sm:space-y-0">
                <div class="flex-1">
                    <div class="relative">
                        <input type="text" 
                               id="logSearch"
                               placeholder="Search logs..." 
                               class="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:ring-1 focus:ring-purple-500 focus:border-purple-500 sm:text-sm">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <svg class="h-5 w-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                                </svg>
                            </div>
                        </div>
                    </div>
                <div class="flex flex-wrap gap-2 sm:ml-4">
                    <select id="logLevel" class="block px-3 py-2 border border-gray-300 rounded-md text-sm leading-5 bg-white focus:outline-none focus:ring-1 focus:ring-purple-500 focus:border-purple-500">
                        <option value="all">All Levels</option>
                        @foreach($levels as $level => $colors)
                            <option value="{{ $level }}">{{ ucfirst($level) }}</option>
                        @endforeach
                    </select>
                    <select id="dateRange" class="block px-3 py-2 border border-gray-300 rounded-md text-sm leading-5 bg-white focus:outline-none focus:ring-1 focus:ring-purple-500 focus:border-purple-500">
                        <option value="today">Today</option>
                        <option value="yesterday">Yesterday</option>
                        <option value="week">Last 7 Days</option>
                        <option value="month">Last 30 Days</option>
                        <option value="all">All Time</option>
                    </select>
                    <button onclick="refreshLogs()" class="inline-flex items-center px-4 py-2 bg-purple-600 text-white text-sm font-medium rounded-md hover:bg-purple-700 transition duration-150 ease-in-out">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                        </svg>
                        Refresh
                    </button>
                </div>
            </div>
        </div>
    </div>
    @endif

    <!-- Logs Container -->
    <div class="bg-white shadow rounded-lg">
        <div class="px-4 py-5 sm:px-6">
            <h3 class="text-lg font-medium text-gray-900">Application Logs</h3>
        </div>
        <div class="border-t border-gray-200">
            <div class="divide-y divide-gray-200">
                @if($logs->count() > 0)
                    @foreach($logs as $log)
                        <div class="px-4 py-3 sm:px-6 hover:bg-gray-50">
                            <div class="flex items-start space-x-3">
                                <div class="flex-shrink-0">
                                    <div class="h-8 w-8 rounded-full flex items-center justify-center
                                        @if($log['level'] === 'error') bg-red-100
                                        @elseif($log['level'] === 'warning') bg-yellow-100
                                        @elseif($log['level'] === 'info') bg-blue-100
                                        @elseif($log['level'] === 'debug') bg-gray-100
                                        @elseif($log['level'] === 'notice') bg-purple-100
                                        @elseif($log['level'] === 'critical') bg-red-200
                                        @elseif($log['level'] === 'alert') bg-orange-200
                                        @elseif($log['level'] === 'emergency') bg-red-300
                                        @else bg-green-100 @endif>
                                        <span class="text-xs font-medium uppercase">
                                            {{ substr($log['level'], 0, 1) }}
                                        </span>
                                    </div>
                                </div>
                                <div class="flex-1 min-w-0">
                                    <div class="text-sm">
                                        <div class="font-mono text-gray-900">{{ $log['message'] }}</div>
                                        @if(isset($log['context']) && $log['context'])
                                            <div class="text-xs text-gray-400 mt-1">Context: {{ $log['context'] }}</div>
                                        @endif
                                    </div>
                                    <div class="text-xs text-gray-500 mt-2">
                                        {{ $log['timestamp']->format('M d, Y H:i:s A') }}
                                    </div>
                                </div>
                            </div>
                        </div>
                    @endforeach
                @else
                    <div class="px-4 py-8 text-center text-gray-500">
                        <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h4a2 2 0 012 2v8a2 2 0 002 2z"></path>
                        </svg>
                        <p class="mt-2">No logs found</p>
                    </div>
                @endif
            </div>
        </div>
    </div>

    @if(!$viewAll)
    <!-- Simple Statistics for Recent View -->
    <div class="mt-6 grid grid-cols-1 md:grid-cols-4 gap-6">
        <div class="bg-white shadow rounded-lg p-5">
            <div class="text-sm font-medium text-gray-500">Recent Logs</div>
            <div class="text-2xl font-semibold text-gray-900">{{ $logs->count() }}</div>
        </div>
        <div class="bg-white shadow rounded-lg p-5">
            <div class="text-sm font-medium text-gray-500">Errors</div>
            <div class="text-2xl font-semibold text-red-600">{{ $logs->where('level', 'error')->count() }}</div>
        </div>
        <div class="bg-white shadow rounded-lg p-5">
            <div class="text-sm font-medium text-gray-500">Warnings</div>
            <div class="text-2xl font-semibold text-yellow-600">{{ $logs->where('level', 'warning')->count() }}</div>
        </div>
        <div class="bg-white shadow rounded-lg p-5">
            <div class="text-sm font-medium text-gray-500">Info</div>
            <div class="text-2xl font-semibold text-blue-600">{{ $logs->where('level', 'info')->count() }}</div>
        </div>
    </div>
    @endif
</div>

    <script>
        // Search logs
        document.getElementById('logSearch').addEventListener('input', function() {
            filterLogs();
        });

        // Filter logs by level
        document.getElementById('logLevel').addEventListener('change', function() {
            filterLogs();
        });

        // Filter logs by date range
        document.getElementById('dateRange').addEventListener('change', function() {
            filterLogs();
        });

        function filterLogs() {
            const searchTerm = document.getElementById('logSearch').value.toLowerCase();
            const level = document.getElementById('logLevel').value;
            const range = document.getElementById('dateRange').value;
            const rows = document.querySelectorAll('.divide-y > div');
            
            const now = new Date();
            let startDate;
            
            switch (range) {
                case 'today':
                    startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate());
                    break;
                case 'yesterday':
                    startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate() - 1);
                    break;
                case 'week':
                    startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate() - 7);
                    break;
                case 'month':
                    startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate() - 30);
                    break;
                case 'all':
                    startDate = new Date(now.getFullYear(), 0, 1);
                    break;
                default:
                    startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate());
            }

            rows.forEach(row => {
                if (row.classList.contains('py-8')) return; // Skip "No logs found" message

                const message = row.querySelector('.font-mono').textContent.toLowerCase();
                const levelSpan = row.querySelector('.uppercase');
                const rowLevel = levelSpan ? levelSpan.textContent.toLowerCase() : '';
                const timestamp = row.querySelector('.text-xs.text-gray-500');
                const logDate = timestamp ? new Date(timestamp.textContent) : null;

                const matchesSearch = message.includes(searchTerm);
                const matchesLevel = level === 'all' || rowLevel === level;
                const matchesDate = !logDate || logDate >= startDate;

                if (matchesSearch && matchesLevel && matchesDate) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        }

        // Refresh logs
        function refreshLogs() {
            location.reload();
        }
    </script>
</div>
@endsection
