@extends('layouts.developer')

@section('title', 'Developer Dashboard')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
    <!-- Welcome Header -->
    <div class="mb-8">
        <div class="bg-gradient-to-r from-purple-600 to-purple-700 rounded-xl shadow-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold mb-2">Developer Dashboard</h1>
                    <p class="text-purple-100">Monitor tenant activities, applications, and system performance</p>
                </div>
                <div class="text-right">
                    <div class="text-sm text-purple-100">System Status</div>
                    <div class="text-lg font-semibold">🟢 Online</div>
                </div>
            </div>
        </div>
    </div>

    <!-- Stats Grid -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <!-- Total Tenants -->
        <div class="bg-white overflow-hidden shadow-lg rounded-xl hover:shadow-xl transition-shadow duration-300">
            <div class="p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="h-12 w-12 bg-gradient-to-r from-blue-500 to-blue-600 rounded-xl flex items-center justify-center shadow-lg">
                            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m-1 4h1m-6-4h1m-1 4h1m-6-4h1m-1 4h1"></path>
                            </svg>
                        </div>
                    </div>
                    <div class="ml-5 w-0 flex-1">
                        <dl>
                            <dt class="text-sm font-medium text-gray-500 truncate">Total Tenants</dt>
                            <dd class="text-2xl font-bold text-gray-900">{{ number_format($tenantStats['total']) }}</dd>
                        </dl>
                    </div>
                </div>
                <div class="mt-4">
                    <div class="flex items-center text-sm">
                        <span class="text-green-600 font-medium">+{{ $tenantStats['new_this_month'] }}</span>
                        <span class="text-gray-500 ml-2">new this month</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Active Tenants -->
        <div class="bg-white overflow-hidden shadow-lg rounded-xl hover:shadow-xl transition-shadow duration-300">
            <div class="p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="h-12 w-12 bg-gradient-to-r from-green-500 to-green-600 rounded-xl flex items-center justify-center shadow-lg">
                            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                    <div class="ml-5 w-0 flex-1">
                        <dl>
                            <dt class="text-sm font-medium text-gray-500 truncate">Active Tenants</dt>
                            <dd class="text-2xl font-bold text-gray-900">{{ number_format($tenantStats['active']) }}</dd>
                        </dl>
                    </div>
                </div>
                <div class="mt-4">
                    <div class="flex items-center text-sm">
                        <span class="text-green-600 font-medium">{{ round(($tenantStats['active'] / max($tenantStats['total'], 1)) * 100) }}%</span>
                        <span class="text-gray-500 ml-2">active rate</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Total Users -->
        <div class="bg-white overflow-hidden shadow-lg rounded-xl hover:shadow-xl transition-shadow duration-300">
            <div class="p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="h-12 w-12 bg-gradient-to-r from-purple-500 to-purple-600 rounded-xl flex items-center justify-center shadow-lg">
                            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                            </svg>
                        </div>
                    </div>
                    <div class="ml-5 w-0 flex-1">
                        <dl>
                            <dt class="text-sm font-medium text-gray-500 truncate">Total Users</dt>
                            <dd class="text-2xl font-bold text-gray-900">{{ number_format($tenantStats['total_users']) }}</dd>
                        </dl>
                    </div>
                </div>
                <div class="mt-4">
                    <div class="flex items-center text-sm">
                        <span class="text-purple-600 font-medium">{{ round($tenantStats['total_users'] / max($tenantStats['total'], 1), 1) }}</span>
                        <span class="text-gray-500 ml-2">avg per tenant</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Active Sessions -->
        <div class="bg-white overflow-hidden shadow-lg rounded-xl hover:shadow-xl transition-shadow duration-300">
            <div class="p-6">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <div class="h-12 w-12 bg-gradient-to-r from-orange-500 to-orange-600 rounded-xl flex items-center justify-center shadow-lg">
                            <svg class="h-6 w-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
                            </svg>
                        </div>
                    </div>
                    <div class="ml-5 w-0 flex-1">
                        <dl>
                            <dt class="text-sm font-medium text-gray-500 truncate">Active Sessions</dt>
                            <dd class="text-2xl font-bold text-gray-900">{{ number_format($tenantStats['active_sessions']) }}</dd>
                        </dl>
                    </div>
                </div>
                <div class="mt-4">
                    <div class="flex items-center text-sm">
                        <span class="text-orange-600 font-medium">Live</span>
                        <span class="text-gray-500 ml-2">chat sessions</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Activities & System Status -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <!-- Active Database Queries -->
        <div class="bg-white shadow-lg rounded-xl" x-data="dashboardQueries()">
            <div class="px-6 py-4 border-b border-gray-200 flex items-center justify-between">
                <div class="flex items-center gap-4">
                    <h3 class="text-lg font-semibold text-gray-900">Active Database Queries</h3>
                    <select 
                        x-model="selectedTenant"
                        @change="updateQueries()"
                        class="text-sm border-gray-300 rounded-md focus:ring-purple-500 focus:border-purple-500 py-1"
                    >
                        <option value="system">System Wide</option>
                        @foreach($tenantStats['list'] ?? [] as $tenant)
                            <option value="{{ $tenant->id }}">{{ $tenant->name }}</option>
                        @endforeach
                    </select>
                </div>
                <a :href="'{{ route('developer.monitoring') }}?view=queries&tenant_id=' + selectedTenant" class="text-sm text-purple-600 hover:text-purple-800 font-medium">Full Monitor</a>
            </div>
            <div class="p-6">
                <div class="space-y-4 relative min-h-[200px]">
                    <!-- Loading Overlay -->
                    <div x-show="loading" class="absolute inset-0 bg-white bg-opacity-60 flex items-center justify-center z-10 rounded-lg">
                        <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-purple-600"></div>
                    </div>

                    <template x-if="queries.length > 0">
                        <template x-for="query in queries" :key="query.ID">
                            <div class="p-4 border border-red-100 rounded-lg hover:bg-red-50 transition duration-150">
                                <div class="flex items-center justify-between mb-2">
                                    <span class="text-xs font-mono font-bold bg-red-50 text-red-700 border border-red-100 px-2 py-1 rounded" x-text="'ID: ' + query.ID"></span>
                                    <span class="text-xs text-red-600 font-bold" x-text="query.TIME + 's'"></span>
                                </div>
                                <div class="text-sm font-mono text-red-700 break-all bg-white border border-red-200 p-3 rounded mb-2 max-h-24 overflow-y-auto shadow-sm" x-text="query.INFO || 'No query info'"></div>
                                <div class="flex items-center gap-4 text-xs text-gray-500">
                                    <span :title="query.DB || 'system'" class="flex items-center gap-1">🗄️ <span class="text-red-600 font-medium" x-text="query.DB || 'system'"></span></span>
                                    <span title="User" class="flex items-center gap-1">👤 <span class="text-red-500" x-text="query.USER"></span></span>
                                    <span title="State" class="flex items-center gap-1">📊 <span x-text="query.STATE || 'Executing'"></span></span>
                                </div>
                            </div>
                        </template>
                    </template>

                    <template x-if="queries.length === 0 && !loading">
                        <div class="text-center py-8">
                            <div class="text-4xl mb-2">⚡</div>
                            <p class="text-sm text-gray-500">No active long-running queries</p>
                        </div>
                    </template>
                </div>
            </div>
        </div>

        <script>
            function dashboardQueries() {
                return {
                    selectedTenant: '{{ $selectedTenantId ?? 'system' }}',
                    queries: @json($activeQueries),
                    loading: false,
                    refreshInterval: null,

                    init() {
                        this.startPolling();
                    },

                    startPolling() {
                        this.refreshInterval = setInterval(() => {
                            this.updateQueries(true);
                        }, 10000); // Refresh every 10 seconds on dashboard
                    },

                    async updateQueries(quiet = false) {
                        if (!quiet) this.loading = true;
                        try {
                            const response = await fetch(`{{ route('developer.monitoring.data') }}?tenant_id=${this.selectedTenant}`);
                            const result = await response.json();
                            if (result.success) {
                                this.queries = result.data.active_queries.slice(0, 5);
                            }
                        } catch (error) {
                            console.error('Failed to update dashboard queries:', error);
                        } finally {
                            if (!quiet) this.loading = false;
                        }
                    }
                }
            }
        </script>

        <!-- Recent Activities -->
        <div class="bg-white shadow-lg rounded-xl">
            <div class="px-6 py-4 border-b border-gray-200">
                <h3 class="text-lg font-semibold text-gray-900">Recent Activities</h3>
            </div>
            <div class="p-6">
                <div class="space-y-4">
                    @forelse ($recentActivities as $activity)
                        <div class="flex items-start space-x-3">
                            <div class="flex-shrink-0">
                                <div class="h-8 w-8 rounded-full flex items-center justify-center
                                    @if($activity['color'] === 'green') bg-green-100 text-green-600
                                    @elseif($activity['color'] === 'blue') bg-blue-100 text-blue-600
                                    @elseif($activity['color'] === 'purple') bg-purple-100 text-purple-600
                                    @elseif($activity['color'] === 'indigo') bg-indigo-100 text-indigo-600
                                    @else bg-gray-100 text-gray-600 @endif">
                                    @if($activity['icon'] === 'user-plus')
                                        <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18 9v3m0 0v3m0 0h3m-3 0h3m-3 0v3m0 0h3m-6-3h3m3 0v3m0 0h-3m-3 0v3m0 0h3"></path>
                                        </svg>
                                    @elseif($activity['icon'] === 'credit-card')
                                        <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1M4 15h2m5-4v1m-1-1h-1m1-1v1m-1 1h-1m-6-4h1m1 1v1m-1 1h-1m1-1v-1m1 1h-1"></path>
                                        </svg>
                                    @elseif($activity['icon'] === 'message-circle')
                                        <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
                                        </svg>
                                    @else
                                        <svg class="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7h-14z"></path>
                                        </svg>
                                    @endif
                                </div>
                            </div>
                            <div class="flex-1 min-w-0">
                                <p class="text-sm text-gray-900">{{ $activity['message'] }}</p>
                                <p class="text-xs text-gray-500">{{ $activity['tenant_name'] }}</p>
                                <p class="text-xs text-gray-400">{{ $activity['timestamp']->diffForHumans() }}</p>
                            </div>
                        </div>
                    @empty
                        <div class="text-center py-8">
                            <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <p class="mt-2 text-sm text-gray-500">No recent activities</p>
                        </div>
                    @endforelse
                </div>
            </div>
        </div>

        <!-- System Status -->
        <div class="bg-white shadow-lg rounded-xl">
            <div class="px-6 py-4 border-b border-gray-200">
                <h3 class="text-lg font-semibold text-gray-900">System Status</h3>
            </div>
            <div class="p-6">
                <div class="space-y-4">
                    <!-- Database Status -->
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <div class="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
                                <svg class="h-4 w-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 7v10c0 2.21 3.582 4 8 4s8-1.79 8-4V7M4 7l8-4m0 0l8 4m0-4l-8 4"></path>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-gray-900">Database</p>
                                <p class="text-xs text-gray-500">{{ $databaseInfo['connection']['database'] }}</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                Online
                            </span>
                        </div>
                    </div>

                    <!-- Cache Status -->
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <div class="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
                                <svg class="h-4 w-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-gray-900">Cache</p>
                                <p class="text-xs text-gray-500">Redis / File</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                Active
                            </span>
                        </div>
                    </div>

                    <!-- Queue Status -->
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <div class="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
                                <svg class="h-4 w-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"></path>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-gray-900">Queue</p>
                                <p class="text-xs text-gray-500">0 jobs pending</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                Ready
                            </span>
                        </div>
                    </div>

                    <!-- Storage Status -->
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <div class="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
                                <svg class="h-4 w-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4"></path>
                                </svg>
                            </div>
                            <div class="ml-3">
                                <p class="text-sm font-medium text-gray-900">Storage</p>
                                <p class="text-xs text-gray-500">{{ $databaseInfo['size'] }} MB</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                Normal
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Links -->
    <div class="bg-white shadow-lg rounded-xl">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg font-semibold text-gray-900">Quick Links</h3>
        </div>
        <div class="p-6">
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <a href="{{ route('developer.tenants') }}" class="group block p-4 border border-gray-200 rounded-lg hover:border-purple-500 hover:shadow-md transition-all duration-200">
                    <div class="flex items-center">
                        <div class="h-10 w-10 bg-purple-100 rounded-lg flex items-center justify-center group-hover:bg-purple-200 transition-colors duration-200">
                            <svg class="h-5 w-5 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m-1 4h1m-6-4h1m-1 4h1m-6-4h1m-1 4h1"></path>
                            </svg>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-gray-900">Manage Tenants</p>
                            <p class="text-xs text-gray-500">View all tenants</p>
                        </div>
                    </div>
                </a>

                <a href="{{ route('developer.database') }}" class="group block p-4 border border-gray-200 rounded-lg hover:border-purple-500 hover:shadow-md transition-all duration-200">
                    <div class="flex items-center">
                        <div class="h-10 w-10 bg-blue-100 rounded-lg flex items-center justify-center group-hover:bg-blue-200 transition-colors duration-200">
                            <svg class="h-5 w-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 7v10c0 2.21 3.582 4 8 4s8-1.79 8-4V7M4 7l8-4m0 0l8 4m0-4l-8 4"></path>
                            </svg>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-gray-900">Database Info</p>
                            <p class="text-xs text-gray-500">View database stats</p>
                        </div>
                    </div>
                </a>

                <a href="{{ route('developer.logs') }}" class="group block p-4 border border-gray-200 rounded-lg hover:border-purple-500 hover:shadow-md transition-all duration-200">
                    <div class="flex items-center">
                        <div class="h-10 w-10 bg-orange-100 rounded-lg flex items-center justify-center group-hover:bg-orange-200 transition-colors duration-200">
                            <svg class="h-5 w-5 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.707.293L19.586 16.707A2 2 0 0120 15.172V8a2 2 0 00-2-2z"></path>
                            </svg>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-gray-900">System Logs</p>
                            <p class="text-xs text-gray-500">View application logs</p>
                        </div>
                    </div>
                </a>

                <a href="{{ route('developer.chat.index') }}" class="group block p-4 border border-gray-200 rounded-lg hover:border-purple-500 hover:shadow-md transition-all duration-200">
                    <div class="flex items-center">
                        <div class="h-10 w-10 bg-green-100 rounded-lg flex items-center justify-center group-hover:bg-green-200 transition-colors duration-200">
                            <svg class="h-5 w-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
                            </svg>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-gray-900">Chat Monitor</p>
                            <p class="text-xs text-gray-500">View user chats</p>
                        </div>
                    </div>
                </a>

                <a href="{{ route('developer.server.config') }}" class="group block p-4 border border-gray-200 rounded-lg hover:border-purple-500 hover:shadow-md transition-all duration-200">
                    <div class="flex items-center">
                        <div class="h-10 w-10 bg-red-100 rounded-lg flex items-center justify-center group-hover:bg-red-200 transition-colors duration-200">
                            <svg class="h-5 w-5 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"></path>
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            </svg>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-gray-900">Server Config</p>
                            <p class="text-xs text-gray-500">Manage environment</p>
                        </div>
                    </div>
                </a>
            </div>
        </div>
    </div>
</div>
@endsection
