# Developer Dashboard Documentation

## Overview

The Developer Dashboard is a comprehensive management interface designed specifically for developers to monitor and manage the SaaS application. It provides insights into tenant activities, application logs, database performance, and system health.

## Features

### 1. Developer Role & Permissions

#### Role Definition
- **Role Name:** `developer`
- **Guard:** `web`
- **Access Level:** Developer-specific dashboard and tools

#### Permissions Created
- `developer.dashboard.view` - Access to main dashboard
- `developer.tenants.view` - View tenant information
- `developer.tenants.activity` - View tenant activities
- `developer.tenants.applications` - View tenant application logs
- `developer.database.view` - View database information
- `developer.logs.view` - View system logs
- `developer.accounts.create` - Create developer accounts
- `developer.roles.manage` - Manage developer roles

#### Developer Accounts
- **Primary Developer:** `developer@example.com` / `developer123`
- **Senior Developer:** `senior.dev@example.com` / `senior123`
- **Backend Developer:** `backend.dev@example.com` / `backend123`
- **Frontend Developer:** `frontend.dev@example.com` / `frontend123`

### 2. Dashboard Components

#### Main Dashboard (`/developer/dashboard`)
- **Tenant Statistics:** Total, active, trial, expired tenants
- **Recent Activities:** Latest tenant activities with timestamps
- **System Status:** Database, cache, queue, storage status
- **Quick Links:** Direct access to key developer tools

#### Tenant Management (`/developer/tenants`)
- **Tenant List:** All tenants with pagination
- **Search & Filter:** Find tenants by name, email, status
- **Tenant Details:** View tenant information and statistics
- **Activity Access:** Direct links to tenant activities and logs

#### Tenant Activity (`/developer/tenants/{id}/activity`)
- **Activity Timeline:** Chronological view of tenant activities
- **Subscription Events:** Subscription creation, changes, cancellations
- **Chat Sessions:** Chat activity and message counts
- **User Management:** User creation and login events

#### Tenant Applications (`/developer/tenants/{id}/applications`)
- **Application Logs:** Tenant-specific application logs
- **Log Filtering:** Filter by log level (error, warning, info, debug)
- **Database Info:** Connection details and table information
- **Log Export:** Download logs for analysis

#### Database Information (`/developer/database`)
- **Database Overview**: Size, table count, and connection status cards.
- **Database Actions**: (Relocated to 2nd row) Optimize, Analyze, Clear Cache, Export Schema, Backup, and SQL Upload.
- **Table Explorer**: Side-by-side view of table list and details.
- **Structure View**: View column types, nullability, keys, and comments.
- **Data Manager**: 
    - **Inline Editing**: Double-click (or Edit button) to modify row data directly.
    - **Add Row**: Dynamic side form for creating new records with automatic field detection.
    - **Delete Row**: Quick removal of records with confirmation.
    - **Pagination**: Efficient browsing of large datasets.
- **Tenant Awareness**: Full support for switching between Main and Tenant databases for all operations.

#### System Logs (`/developer/logs`)
- **Application Logs**: Laravel application logs with improved error handling and real-time filtering.
- **Search & Filter**: 
    - **Real-time Search**: Instant searching through log messages.
    - **Level Filtering**: Filter logs by PSR-3 levels (Error, Warning, Info, etc.).
    - **Date Range**: Filter logs by predefined ranges (Today, Yesterday, Last 7 Days, Last 30 Days, All Time).
- **Log Management**:
    - **Download**: Export current logs as a text file for offline analysis.
    - **Clear Logs**: Securely truncate the log file with a confirmation dialog.
- **Robustness**: Fixed common `Undefined array key` errors in log parsing and JavaScript syntax issues in filtering.

#### Server Configuration (`/developer/server-config`)
- **Environment Management**: View and update `.env` variables securely from the dashboard.
- **Cache Management**: Clear application, config, route, and view caches in one click.
- **Artisan Command Runner**: Execute permitted artisan commands directly from the UI.
- **Server Info**: View PHP version, Laravel version, and server environment details.

#### App Monitoring (`/developer/monitoring`)
- **Real-time Performance**: Live dashboard for system and tenant-specific metrics.
- **Dynamic Selection**: Switch between System-wide and individual Tenant views.
- **Tenant User Insights**: Select a specific user within a tenant to view their profile details alongside tenant metrics.
- **Live Metrics**:
    - **System**: PHP/Laravel versions, memory usage, tenant distribution stats, and system-wide active queries.
    - **Tenant**: Status, database size, table count, user activity (last 24h), and tenant-specific queries.
- **Process Monitoring**: Real-time view of active database queries (ID, Time, User, Host, Query info) with auto-refresh.

#### Dashboard Overview (`/developer`)
- **Consolidated Stats**: Quick view of total tenants, active users, and system health.
- **Active Queries Widget**: A high-level overview of currently executing database processes, accessible directly from the main dashboard.
- **Navigation**: Dedicated sidebar links for **App Monitoring**, **Active Queries**, and **System Logs**.
- **Live Logs**: Quick access to recent application logs with level-based styling.
- **Auto-Refresh**: Configurable automatic data updates every 30 seconds.

## Technical Implementation

### File Structure

```
app/Http/Controllers/Developer/
├── DashboardController.php          # Main controller
resources/views/developer/
├── dashboard/
│   └── index.blade.php              # Main dashboard
├── tenants/
│   ├── index.blade.php              # Tenant list
│   ├── activity.blade.php           # Tenant activity
│   └── applications.blade.php       # Tenant logs
├── database/
│   └── index.blade.php              # Database info
├── logs/
│   └── index.blade.php              # System logs
├── layouts/
│   └── developer.blade.php          # Developer layout
database/seeders/
└── DeveloperSeeder.php              # Developer role & accounts
```

### Routes

```php
// Developer Routes
Route::prefix('developer')->middleware(['web', 'auth', 'developer'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('developer.dashboard');
    Route::get('/tenants', [DashboardController::class, 'tenants'])->name('developer.tenants');
    Route::post('/tenants/assign-subscription', [DashboardController::class, 'assignSubscription'])->name('developer.tenants.subscription.assign');
    Route::get('/tenants/{id}/activity', [DashboardController::class, 'tenantActivity'])->name('developer.tenants.activity');
    Route::get('/tenants/{id}/applications', [DashboardController::class, 'tenantApplications'])->name('developer.tenants.applications');
    
    // Monitoring
    Route::get('/monitoring', [DashboardController::class, 'monitoring'])->name('developer.monitoring');
    Route::get('/monitoring/data', [DashboardController::class, 'getMonitoringData'])->name('developer.monitoring.data');
    
    // Database
    Route::get('/database', [DashboardController::class, 'database'])->name('developer.database');
    Route::get('/database/tables', [DashboardController::class, 'getTables'])->name('developer.database.tables');
    Route::get('/database/columns', [DashboardController::class, 'getTableColumns'])->name('developer.database.columns');
    Route::post('/database/column/update', [DashboardController::class, 'updateTableColumn'])->name('developer.database.column.update');
    
    // Table Data
    Route::get('/database/data', [DashboardController::class, 'getTableData'])->name('developer.database.data');
    Route::post('/database/data/add', [DashboardController::class, 'addTableRow'])->name('developer.database.data.add');
    Route::post('/database/data/update', [DashboardController::class, 'updateTableRow'])->name('developer.database.data.update');
    Route::post('/database/data/delete', [DashboardController::class, 'deleteTableRow'])->name('developer.database.data.delete');

    // Maintenance
    Route::post('/database/clear-cache', [DashboardController::class, 'clearCache'])->name('developer.database.clear-cache');
    Route::post('/database/optimize', [DashboardController::class, 'optimizeDatabase'])->name('developer.database.optimize');
    Route::post('/database/analyze', [DashboardController::class, 'analyzeQueries'])->name('developer.database.analyze');
    Route::post('/database/export', [DashboardController::class, 'exportSchema'])->name('developer.database.export');
    Route::post('/database/backup', [DashboardController::class, 'backupDatabase'])->name('developer.database.backup');
    Route::post('/database/upload', [DashboardController::class, 'uploadDatabase'])->name('developer.database.upload');

    // System
    Route::get('/logs', [DashboardController::class, 'logs'])->name('developer.logs');
    Route::get('/logs/download', [DashboardController::class, 'downloadLogs'])->name('developer.logs.download');
    Route::post('/logs/clear', [DashboardController::class, 'clearLogs'])->name('developer.logs.clear');
    Route::get('/server-config', [DashboardController::class, 'serverConfig'])->name('developer.server.config');
    Route::post('/server/update-env', [DashboardController::class, 'updateEnv'])->name('developer.server.update-env');
    Route::post('/server/clear-cache', [DashboardController::class, 'clearAllCaches'])->name('developer.server.clear-cache');
    Route::post('/server/run-command', [DashboardController::class, 'runArtisanCommand'])->name('developer.server.run-command');

    // Shared Admin/Developer
    Route::get('/chat', [\App\Http\Controllers\Admin\ChatController::class, 'index'])->name('developer.chat.index');
    Route::get('/analytics', [\App\Http\Controllers\Admin\ReportController::class, 'analytics'])->name('developer.reports.analytics');
    
    // User/Role Management
    Route::resource('/users', \App\Http\Controllers\Developer\UserController::class, ['names' => 'developer.users']);
    Route::resource('/roles', \App\Http\Controllers\Developer\RoleController::class, ['names' => 'developer.roles']);
});
```

### Security

#### Middleware
- **Authentication:** `auth` - User must be logged in
- **Authorization:** `permission:developer.dashboard.view` - User must have developer permission
- **Web Middleware:** `web` - Session and CSRF protection

#### Access Control
- Only users with `developer` role can access the dashboard
- Each feature requires specific permission
- Route-level permission checking
- Blade template-level permission checks

### Database Queries

#### Tenant Statistics
```php
$tenantStats = [
    'total' => Tenant::count(),
    'active' => Tenant::where('status', 'active')->count(),
    'trialing' => Tenant::where('status', 'trialing')->count(),
    'expired' => Tenant::where('status', 'expired')->count(),
    'new_this_month' => Tenant::where('created_at', '>=', Carbon::now()->startOfMonth())->count(),
    'with_subscriptions' => Tenant::whereHas('subscription')->count(),
    'total_users' => User::count(),
    'active_sessions' => ChatSession::where('status', 'active')->count(),
];
```

#### Database Information
```php
$databaseSize = DB::select("SELECT table_schema AS 'database', 
    ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS 'size_mb'
    FROM information_schema.tables 
    WHERE table_schema = '" . env('DB_DATABASE') . "'")
    ->first();

$tables = DB::select("SELECT table_name, 
    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS 'size_mb',
    table_rows
    FROM information_schema.tables 
    WHERE table_schema = '" . env('DB_DATABASE') . "'
    ORDER BY (data_length + index_length) DESC")
    ->take(20);
```

## User Interface

### Design System
- **Color Scheme:** Purple theme for developer interface
- **Layout:** Fixed sidebar with main content area
- **Responsive:** Mobile-friendly design
- **Icons:** Heroicons for consistent iconography

### Navigation
- **Dashboard:** Main overview and statistics
- **Tenants:** Tenant management and monitoring
- **Database:** Database information and tools
- **Logs:** System logs and debugging
- **Quick Actions:** Links to admin tools

### Interactive Features
- **Real-time Updates:** Auto-refresh for statistics
- **Search & Filter:** Dynamic filtering of data
- **Export Functionality:** Download logs and data
- **Database Actions:** Optimize, clear cache, export schema

## API Endpoints

### Database Management
- `POST /developer/database/clear-cache` - Clear application cache
- `POST /developer/database/optimize` - Optimize database tables
- `POST /developer/database/analyze` - Analyze slow queries
- `POST /developer/database/export` - Export database schema

### Response Format
```json
{
    "success": true,
    "message": "Operation completed successfully",
    "data": { ... }
}
```

## Installation & Setup

### 1. Run the Developer Seeder
```bash
php artisan db:seed --class=DeveloperSeeder
```

### 2. Login with Developer Account
- URL: `/login`
- Email: `developer@example.com`
- Password: `developer123`

### 3. Access Developer Dashboard
- URL: `/developer/dashboard`

## Usage Examples

### Monitoring Tenant Activity
1. Navigate to `/developer/tenants`
2. Click "Activity" on any tenant
3. View chronological activity timeline
4. Filter by activity type if needed

### Viewing & Managing Application Logs
1. Navigate to `/developer/logs`
2. **Search**: Type in the search box to find specific log entries instantly.
3. **Filter**: Use the dropdowns to filter by log level (Error, Warning, etc.) and date range.
4. **Download**: Click "Download" to save the current log file to your computer.
5. **Clear**: Click "Clear Logs" and confirm to reset the application log file.

### Database Management
1. Navigate to `/developer/database`
2. View database statistics and table sizes
3. Click "Optimize Database" to improve performance
4. Click "Clear Cache" to refresh application cache

### Tenant-Specific Debugging
1. Navigate to `/developer/tenants/{id}/applications`
2. View tenant-specific application logs
3. Check database connection details
4. Export logs for detailed analysis

## Troubleshooting

### Common Issues

#### Permission Denied
- **Problem:** User cannot access developer dashboard
- **Solution:** Ensure user has `developer` role and proper permissions
- **Command:** `php artisan db:seed --class=DeveloperSeeder`

#### Database Information Not Loading
- **Problem:** Database size and table information not showing
- **Solution:** Check database permissions for `information_schema` access
- **Command:** Verify database user has SELECT permissions

#### Logs Not Displaying
- **Problem:** Application logs not appearing
- **Solution:** Check log file permissions and path
- **Command:** Ensure `storage/logs/laravel.log` is readable

#### Cache Clear Not Working
- **Problem:** Cache clear operation fails
- **Solution:** Check file permissions for storage directories
- **Command:** `chmod -R 755 storage/`

### Debug Mode
Enable debug mode for detailed error messages:
```php
// .env
APP_DEBUG=true
```

### Log Level Configuration
Configure log levels in `config/logging.php`:
```php
'channels' => [
    'single' => [
        'driver' => 'single',
        'path' => storage_path('logs/laravel.log'),
        'level' => 'debug',
    ],
],
```

## Security Considerations

### Access Control
- Only authenticated users with developer role can access
- Each feature requires specific permission
- Route-level protection prevents unauthorized access

### Data Protection
- Sensitive database information is limited to developers
- Log files may contain sensitive information
- Export functionality should be used carefully

### Audit Trail
- All developer actions are logged
- Database operations are tracked
- User access is monitored

## Performance Optimization

### Database Queries
- Use database indexes for frequent queries
- Optimize slow queries regularly
- Monitor database size and growth

### Caching
- Clear cache regularly during development
- Use appropriate cache strategies
- Monitor cache hit rates

### Log Management
- Rotate logs regularly to prevent disk space issues
- Filter logs by level to reduce noise
- Archive old logs for compliance

## Future Enhancements

### Planned Features
- **Real-time Monitoring:** WebSocket-based real-time updates
- **Performance Metrics:** Advanced performance monitoring
- **Alert System:** Automated alerts for system issues
- **API Documentation:** Interactive API documentation
- **Debug Tools:** Advanced debugging utilities

### Integration Possibilities
- **External Monitoring:** Integration with monitoring services
- **CI/CD Pipeline:** Deployment pipeline integration
- **Version Control:** Git integration for code management
- **Testing Tools:** Automated testing framework integration

## Conclusion

The Developer Dashboard provides a comprehensive set of tools for monitoring and managing the SaaS application. It offers insights into tenant activities, system performance, and application health, making it an essential tool for developers working on the platform.

The dashboard is designed with security, performance, and usability in mind, providing a professional interface for technical users while maintaining the security and integrity of the system.
