<?php

namespace Database\Seeders\Tenant;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use App\Models\Tenant\Category;
use App\Models\Tenant\Unit;
use App\Models\Tenant\Brand;
use App\Models\Tenant\Manufacturer;
use App\Models\Tenant\Product;
use App\Models\Tenant\Customer;
use App\Models\Tenant\Supplier;
use App\Models\Tenant\Sale;
use App\Models\Tenant\Purchase;
use Illuminate\Support\Str;

class TenantDemoDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // 1. Categories
        $categories = [
            'Fruits & Vegetables',
            'Dairy & Eggs',
            'Bakery',
            'Beverages',
            'Snacks',
            'Meat & Seafood',
            'Pantry Essentials',
            'Household',
            'Personal Care',
            'Frozen Foods'
        ];
        
        $categoryModels = [];
        foreach ($categories as $cat) {
            $categoryModels[] = Category::firstOrCreate(['name' => $cat]);
        }

        // 2. Units
        $units = [
            ['name' => 'Kilogram', 'short_name' => 'kg', 'type' => 'base'],
            ['name' => 'Gram', 'short_name' => 'g', 'type' => 'base'],
            ['name' => 'Liter', 'short_name' => 'L', 'type' => 'base'],
            ['name' => 'Milliliter', 'short_name' => 'ml', 'type' => 'base'],
            ['name' => 'Piece', 'short_name' => 'pc', 'type' => 'base'],
            ['name' => 'Pack', 'short_name' => 'pk', 'type' => 'base'],
            ['name' => 'Box', 'short_name' => 'bx', 'type' => 'base'],
            ['name' => 'Dozen', 'short_name' => 'dz', 'type' => 'base'],
        ];

        $unitModels = [];
        foreach ($units as $unit) {
            $unitModels[$unit['short_name']] = Unit::firstOrCreate(['short_name' => $unit['short_name']], $unit);
        }

        // 3. Brands
        $brands = ['Farm Fresh', 'Organic Valley', 'Coca-Cola', 'Pepsi', 'Lays', 'Nestle', 'Kelloggs', 'Unilever', 'P&G', 'Local Farms'];
        $brandModels = [];
        foreach ($brands as $brand) {
            $brandModels[] = Brand::firstOrCreate(['name' => $brand]);
        }

        // 4. Manufacturers
        $manufacturers = ['Global Foods Inc.', 'Local Distributors', 'Fresh Harvest Co.', 'Beverage Giants'];
        $manufacturerModels = [];
        foreach ($manufacturers as $man) {
            $manufacturerModels[] = Manufacturer::firstOrCreate(['name' => $man, 'contact_person' => 'Manager ' . $man]);
        }

        // 5. Suppliers
        $suppliers = [];
        for ($i = 1; $i <= 5; $i++) {
            $suppliers[] = Supplier::firstOrCreate(
                ['email' => "supplier{$i}@example.com"],
                [
                    'name' => "Supplier {$i} Wholesale",
                    'contact' => "Agent {$i} - 555-010{$i}",
                ]
            );
        }

        // 6. Customers
        $customers = [];
        for ($i = 1; $i <= 10; $i++) {
            $customers[] = Customer::firstOrCreate(
                ['email' => "customer{$i}@example.com"],
                [
                    'name' => "Customer {$i}",
                    'contact' => "555-020{$i}",
                ]
            );
        }

        // 7. Products
        $products = [
            [
                'name' => 'Red Apple',
                'category' => 'Fruits & Vegetables',
                'brand' => 'Farm Fresh',
                'unit' => 'kg',
                'price' => 3.50,
                'cost' => 2.00,
                'stock' => 100
            ],
            [
                'name' => 'Banana',
                'category' => 'Fruits & Vegetables',
                'brand' => 'Local Farms',
                'unit' => 'kg',
                'price' => 1.20,
                'cost' => 0.80,
                'stock' => 150
            ],
            [
                'name' => 'Whole Milk',
                'category' => 'Dairy & Eggs',
                'brand' => 'Organic Valley',
                'unit' => 'L',
                'price' => 1.50,
                'cost' => 1.00,
                'stock' => 50
            ],
            [
                'name' => 'Eggs (Dozen)',
                'category' => 'Dairy & Eggs',
                'brand' => 'Local Farms',
                'unit' => 'dz',
                'price' => 4.00,
                'cost' => 3.00,
                'stock' => 60
            ],
            [
                'name' => 'Coca-Cola Can',
                'category' => 'Beverages',
                'brand' => 'Coca-Cola',
                'unit' => 'pc',
                'price' => 1.00,
                'cost' => 0.60,
                'stock' => 200
            ],
            [
                'name' => 'Potato Chips',
                'category' => 'Snacks',
                'brand' => 'Lays',
                'unit' => 'pk',
                'price' => 2.50,
                'cost' => 1.50,
                'stock' => 80
            ],
            [
                'name' => 'Whole Wheat Bread',
                'category' => 'Bakery',
                'brand' => 'Local Farms',
                'unit' => 'pc',
                'price' => 3.00,
                'cost' => 1.80,
                'stock' => 40
            ],
            [
                'name' => 'Chicken Breast',
                'category' => 'Meat & Seafood',
                'brand' => 'Local Farms',
                'unit' => 'kg',
                'price' => 8.00,
                'cost' => 6.00,
                'stock' => 30
            ],
            [
                'name' => 'Rice (5kg Bag)',
                'category' => 'Pantry Essentials',
                'brand' => 'Global Foods Inc.',
                'unit' => 'pc',
                'price' => 12.00,
                'cost' => 9.00,
                'stock' => 20
            ],
            [
                'name' => 'Shampoo',
                'category' => 'Personal Care',
                'brand' => 'Unilever',
                'unit' => 'pc',
                'price' => 5.50,
                'cost' => 3.50,
                'stock' => 25
            ]
        ];

        $productVariants = [];

        foreach ($products as $p) {
            // Find the category and brand from the models we already created
            $cat = collect($categoryModels)->firstWhere('name', $p['category']);
            $brand = collect($brandModels)->firstWhere('name', $p['brand']);
            $unit = $unitModels[$p['unit']];

            if (!$cat || !$brand) {
                continue; // Skip if category or brand not found
            }

            $product = Product::firstOrCreate(
                ['name' => $p['name']],
                [
                    'category_id' => $cat->id,
                    'brand_id' => $brand->id,
                    'unit_id' => $unit->id,
                    'has_variants' => false,
                ]
            );

            $variant = $product->variants()->firstOrCreate(
                ['sku' => Str::slug($p['name']) . '-001'],
                [
                    'name' => $p['name'],
                    'price' => $p['price'],
                    'cost' => $p['cost'],
                    'stock' => $p['stock'],
                ]
            );
            
            $productVariants[] = $variant;
        }

        // 8. Purchases (Restocking)
        foreach ($suppliers as $supplier) {
            $purchase = Purchase::create([
                'supplier_id' => $supplier->id,
                'status' => 'received',
                'total' => 0,
            ]);

            $total = 0;
            // Add 3-5 random items
            for ($k = 0; $k < rand(3, 5); $k++) {
                $variant = $productVariants[array_rand($productVariants)];
                $qty = rand(10, 50);
                $cost = $variant->cost;
                $lineTotal = $qty * $cost;

                $purchase->items()->create([
                    'variant_id' => $variant->id,
                    'quantity' => $qty,
                    'cost' => $cost,
                    'total' => $lineTotal,
                ]);

                $total += $lineTotal;
            }
            $purchase->update(['total' => $total]);
        }

        // 9. Sales (Transactions)
        // Generate 20 sales over the last 30 days
        for ($i = 0; $i < 20; $i++) {
            $customer = $customers[array_rand($customers)];
            
            $sale = Sale::create([
                'customer_id' => $customer->id,
                'subtotal' => 0,
                'discount' => 0,
                'total' => 0,
                'paid_amount' => 0,
                'payment_method' => ['cash', 'card'][rand(0, 1)],
                'status' => 'completed',
                'payment_status' => 'paid',
            ]);

            $subtotal = 0;
            // Add 1-5 random items
            for ($j = 0; $j < rand(1, 5); $j++) {
                $variant = $productVariants[array_rand($productVariants)];
                $qty = rand(1, 5);
                $price = $variant->price;
                $lineTotal = $qty * $price;

                $sale->items()->create([
                    'variant_id' => $variant->id,
                    'quantity' => $qty,
                    'price' => $price,
                    'total' => $lineTotal,
                ]);

                $subtotal += $lineTotal;
            }
            
            $total = $subtotal; // Simplified - no tax calculation for now

            $sale->update([
                'subtotal' => $subtotal,
                'total' => $total,
                'paid_amount' => $total,
            ]);
        }
    }
}
