<?php

namespace Database\Seeders\Tenant;

use Illuminate\Database\Seeder;
use App\Models\Tenant\Product;
use App\Models\Tenant\Category;
use App\Models\Tenant\Brand;
use App\Models\Tenant\Unit;
use App\Models\Tenant\ProductVariant;
use Illuminate\Support\Facades\DB;

class ComplexProductSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $category = Category::firstOrCreate(['name' => 'Beverages']);
        $brand = Brand::firstOrCreate(['name' => 'PepsiCo']);
        $baseUnit = Unit::firstOrCreate(['name' => 'Piece', 'short_name' => 'pc', 'type' => 'base']);

        $productName = 'Pepsi';
        
        // Check if exists
        if (Product::where('name', $productName)->exists()) {
            $this->command->info("Product {$productName} already exists.");
            return;
        }

        DB::transaction(function () use ($productName, $category, $brand, $baseUnit) {
            // 1. Create Product
            $product = Product::create([
                'name' => $productName,
                'category_id' => $category->id,
                'brand_id' => $brand->id,
                'unit_id' => $baseUnit->id,
                'has_variants' => true,
            ]);

            // 2. Create Units
            // Base Unit Wrapper
            $unitMap = [];
            $baseWrapper = $product->productUnits()->create([
                'name' => $baseUnit->name,
                'conversion_factor' => 1,
                'is_base' => true,
                'is_purchase' => true,
                'is_sale' => true,
            ]);
            $unitMap['Piece'] = $baseWrapper->id;

            // Case (24)
            $caseWrapper = $product->productUnits()->create([
                'name' => 'Case',
                'conversion_factor' => 24,
                'is_base' => false,
                'is_purchase' => true,
                'is_sale' => true,
            ]);
            $unitMap['Case'] = $caseWrapper->id;

            // Pack (6)
            $packWrapper = $product->productUnits()->create([
                'name' => 'Pack',
                'conversion_factor' => 6,
                'is_base' => false,
                'is_purchase' => false,
                'is_sale' => true,
            ]);
            $unitMap['Pack'] = $packWrapper->id;

            // 3. Define Variants Data
            $variants = [
                [
                    'sku' => 'PEP-CAN-250',
                    'attrs' => ['Type' => 'Can', 'Size' => '250ml'],
                    'prices' => [
                        'Piece' => ['price' => 0.50, 'cost' => 0.30],
                        'Case' => ['price' => 10.00, 'cost' => 7.00],
                        'Pack' => ['price' => 2.80, 'cost' => 1.80],
                    ]
                ],
                [
                    'sku' => 'PEP-CAN-400',
                    'attrs' => ['Type' => 'Can', 'Size' => '400ml'],
                    'prices' => [
                        'Piece' => ['price' => 0.80, 'cost' => 0.50],
                        'Case' => ['price' => 18.00, 'cost' => 11.50],
                    ]
                ],
                [
                    'sku' => 'PEP-PET-1L',
                    'attrs' => ['Type' => 'Pet Bottle', 'Size' => '1L'],
                    'prices' => [
                        'Piece' => ['price' => 1.20, 'cost' => 0.80],
                        'Case' => ['price' => 25.00, 'cost' => 18.00],
                    ]
                ]
            ];

            foreach ($variants as $vData) {
                // Create Variant
                $variant = $product->variants()->create([
                    'name' => $productName,
                    'sku' => $vData['sku'],
                    'stock' => 100,
                    'is_active' => true,
                ]);

                // Attributes
                foreach ($vData['attrs'] as $name => $value) {
                    $variant->attributes()->create(['name' => $name, 'value' => $value]);
                }

                // Prices
                foreach ($vData['prices'] as $unitName => $priceData) {
                    if (isset($unitMap[$unitName])) {
                        $variant->prices()->create([
                            'product_unit_id' => $unitMap[$unitName],
                            'price' => $priceData['price'],
                            'cost' => $priceData['cost'],
                        ]);
                    }
                }
            }
        });

        $this->command->info("Created product: {$productName} with complex variants.");
    }
}
