<?php

namespace Database\Seeders\Tenant;

use Illuminate\Database\Seeder;
use App\Models\Tenant\Product;
use App\Models\Tenant\Category;
use App\Models\Tenant\Brand;
use App\Models\Tenant\Unit;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class BulkProductSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $count = 100;
        if ($this->command) {
            $this->command->info("Seeding {$count} products...");
        } else {
            echo "Seeding {$count} products...\n";
        }

        $categories = ['Beverages', 'Snacks', 'Dairy', 'Vegetables', 'Fruits', 'Bakery', 'Meat', 'Household', 'Personal Care'];
        $brands = ['Nestle', 'Coca-Cola', 'PepsiCo', 'Unilever', 'P&G', 'Kraft', 'Kellogg', 'General Mills', 'Danone', 'Local Farm'];
        $units = [
            ['name' => 'Piece', 'short' => 'pc'],
            ['name' => 'Kilogram', 'short' => 'kg'],
            ['name' => 'Liter', 'short' => 'L'],
            ['name' => 'Box', 'short' => 'box'],
        ];

        // Ensure base data exists
        $catIds = [];
        foreach ($categories as $cat) {
            $catIds[] = Category::firstOrCreate(['name' => $cat])->id;
        }

        $brandIds = [];
        foreach ($brands as $brand) {
            $brandIds[] = Brand::firstOrCreate(['name' => $brand])->id;
        }

        $unitIds = [];
        foreach ($units as $u) {
            $unitIds[] = Unit::firstOrCreate(
                ['name' => $u['name']],
                ['short_name' => $u['short'], 'type' => 'base']
            )->id;
        }

        $adjectives = ['Fresh', 'Organic', 'Premium', 'Classic', 'Spicy', 'Sweet', 'Frozen', 'Natural', 'Tasty', 'Super'];
        $nouns = ['Apple', 'Banana', 'Milk', 'Bread', 'Chicken', 'Soap', 'Juice', 'Chips', 'Coffee', 'Tea', 'Rice', 'Pasta', 'Cheese', 'Yogurt', 'Butter', 'Water', 'Soda', 'Biscuits', 'Cookies', 'Cake'];

        for ($i = 0; $i < $count; $i++) {
            $name = $adjectives[array_rand($adjectives)] . ' ' . $nouns[array_rand($nouns)] . ' ' . rand(1, 999);
            $catId = $catIds[array_rand($catIds)];
            $brandId = $brandIds[array_rand($brandIds)];
            $unitId = $unitIds[array_rand($unitIds)];
            
            // Randomly decide if complex (20% chance)
            $isComplex = rand(1, 100) <= 20;

            try {
                DB::transaction(function () use ($name, $catId, $brandId, $unitId, $isComplex) {
                    $product = Product::create([
                        'name' => $name,
                        'category_id' => $catId,
                        'brand_id' => $brandId,
                        'unit_id' => $unitId,
                        'has_variants' => $isComplex,
                        'image' => null, // Could add placeholder images if needed
                    ]);

                    $baseUnit = Unit::find($unitId);
                    
                    // Always create Base Unit Wrapper
                    $baseWrapper = $product->productUnits()->create([
                        'name' => $baseUnit->name,
                        'conversion_factor' => 1,
                        'is_base' => true,
                        'is_purchase' => true,
                        'is_sale' => true,
                    ]);

                    if (!$isComplex) {
                        // Simple Product
                        $price = rand(100, 5000) / 100; // 1.00 to 50.00
                        $cost = $price * 0.7; // 30% margin
                        
                        $product->variants()->create([
                            'name' => $name,
                            'sku' => strtoupper(Str::slug($name)) . '-' . rand(1000, 9999),
                            'price' => $price,
                            'cost' => $cost,
                            'stock' => rand(10, 500),
                            'is_active' => true,
                        ]);
                    } else {
                        // Complex Product
                        // Create a secondary unit (e.g., Box of 12)
                        $boxWrapper = $product->productUnits()->create([
                            'name' => 'Box',
                            'conversion_factor' => 12,
                            'is_base' => false,
                            'is_purchase' => true,
                            'is_sale' => true,
                        ]);

                        // Create 2-3 variants (e.g., Colors or Sizes)
                        $sizes = ['Small', 'Medium', 'Large'];
                        $colors = ['Red', 'Blue', 'Green'];
                        
                        $variantCount = rand(2, 3);
                        for ($v = 0; $v < $variantCount; $v++) {
                            $size = $sizes[$v];
                            $color = $colors[array_rand($colors)];
                            
                            $sku = strtoupper(Str::slug($name)) . '-' . strtoupper(substr($size, 0, 1)) . '-' . rand(100, 999);
                            
                            $variant = $product->variants()->create([
                                'name' => $name . " ($size, $color)", // This might be dynamically built in UI, but good for backend
                                'sku' => $sku,
                                'stock' => rand(10, 200),
                                'is_active' => true,
                            ]);

                            // Attributes
                            $variant->attributes()->create(['name' => 'Size', 'value' => $size]);
                            $variant->attributes()->create(['name' => 'Color', 'value' => $color]);

                            // Prices for Base Unit
                            $basePrice = rand(100, 5000) / 100;
                            $variant->prices()->create([
                                'product_unit_id' => $baseWrapper->id,
                                'price' => $basePrice,
                                'cost' => $basePrice * 0.7,
                            ]);

                            // Prices for Box Unit
                            $boxPrice = $basePrice * 12 * 0.9; // 10% discount for box
                            $variant->prices()->create([
                                'product_unit_id' => $boxWrapper->id,
                                'price' => $boxPrice,
                                'cost' => $boxPrice * 0.7,
                            ]);
                        }
                    }
                });
            } catch (\Exception $e) {
                if ($this->command) {
                    $this->command->error("Failed to seed product {$name}: " . $e->getMessage());
                } else {
                    echo "Failed to seed product {$name}: " . $e->getMessage() . "\n";
                }
            }
        }
        
        if ($this->command) {
            $this->command->info("Seeding completed!");
        } else {
            echo "Seeding completed!\n";
        }
    }
}
