<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // 1. Modify products table
        if (!Schema::hasColumn('products', 'has_variants')) {
            Schema::table('products', function (Blueprint $table) {
                $table->boolean('has_variants')->default(false)->after('type');
            });
        }

        // 2. Modify product_variants table
        if (!Schema::hasColumn('product_variants', 'is_active')) {
            Schema::table('product_variants', function (Blueprint $table) {
                $table->boolean('is_active')->default(true)->after('stock');
            });
        }

        // 3. Create variant_attributes table
        Schema::create('variant_attributes', function (Blueprint $table) {
            $table->id();
            $table->foreignId('product_variant_id')->constrained('product_variants')->onDelete('cascade');
            $table->string('name'); // e.g., 'Color', 'Size'
            $table->string('value'); // e.g., 'Red', 'XL'
            $table->timestamps();
        });

        // 4. Create product_units table (For multi-unit handling like Case, Pack)
        Schema::create('product_units', function (Blueprint $table) {
            $table->id();
            $table->foreignId('product_id')->constrained('products')->onDelete('cascade');
            $table->string('name'); // e.g., 'Case', 'Pack', 'Piece'
            $table->decimal('conversion_factor', 10, 4)->default(1); // e.g., 24 (if base is piece)
            $table->boolean('is_base')->default(false);
            $table->boolean('is_purchase')->default(true);
            $table->boolean('is_sale')->default(true);
            $table->timestamps();
        });

        // 5. Create variant_prices table (Price per unit per variant)
        Schema::create('variant_prices', function (Blueprint $table) {
            $table->id();
            $table->foreignId('product_variant_id')->constrained('product_variants')->onDelete('cascade');
            $table->foreignId('product_unit_id')->constrained('product_units')->onDelete('cascade');
            $table->decimal('price', 10, 2)->default(0);
            $table->decimal('cost', 10, 2)->default(0);
            $table->integer('stock_alert')->nullable(); // Optional: Alert level for this specific unit packaging if needed
            $table->timestamps();
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('variant_prices');
        Schema::dropIfExists('product_units');
        Schema::dropIfExists('variant_attributes');
        
        if (Schema::hasColumn('product_variants', 'is_active')) {
            Schema::table('product_variants', function (Blueprint $table) {
                $table->dropColumn('is_active');
            });
        }

        if (Schema::hasColumn('products', 'has_variants')) {
            Schema::table('products', function (Blueprint $table) {
                $table->dropColumn('has_variants');
            });
        }
    }
};
