<?php

namespace App\Services;

use App\Models\Tenant;
use App\Models\TenantCategory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;

class TenantSyncService
{
    /**
     * Sync default categories from SaaS Admin to a specific tenant database.
     */
    public function syncCategoriesToTenant(Tenant $tenant, array $defaultCategories)
    {
        if (empty($defaultCategories)) {
            return;
        }

        try {
            // Configure tenant connection
            Config::set('database.connections.tenant.database', $tenant->database_name);
            DB::purge('tenant');
            
            // Reconnect and start transaction in tenant DB
            DB::connection('tenant')->beginTransaction();

            foreach ($defaultCategories as $catName) {
                if (empty($catName)) continue;

                DB::connection('tenant')->table('categories')->updateOrInsert(
                    ['name' => trim($catName)],
                    [
                        'name' => trim($catName),
                        'slug' => \Illuminate\Support\Str::slug($catName),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]
                );
            }

            DB::connection('tenant')->commit();
            Log::info("Successfully synced categories to tenant: {$tenant->name}");
        } catch (\Exception $e) {
            DB::connection('tenant')->rollBack();
            Log::error("Failed to sync categories to tenant {$tenant->name}: " . $e->getMessage());
        } finally {
            // Clean up connection
            DB::purge('tenant');
        }
    }

    /**
     * Sync a TenantCategory to all its associated tenants.
     */
    public function syncTenantCategoryToAll(TenantCategory $category)
    {
        $defaultCategories = $category->default_categories ?? [];
        
        // Find all tenants that should be in this category
        // Path: Tenant -> Subscription -> Plan -> TenantCategory
        $tenants = Tenant::whereHas('subscription.plan', function ($query) use ($category) {
            $query->where('tenant_category_id', $category->id);
        })->get();

        foreach ($tenants as $tenant) {
            $this->syncCategoriesToTenant($tenant, $defaultCategories);
        }
    }
}
