<?php

namespace App\Services;

use App\Models\AppSetting;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Customer as StripeCustomer;

class PaymentService
{
    protected $gateway;
    protected $config;

    public function __construct()
    {
        $this->gateway = AppSetting::get('payment_gateway', 'stripe');
        $this->loadConfig();
    }

    protected function loadConfig()
    {
        $this->config = [
            'stripe' => [
                'public_key' => AppSetting::get('payment_key'),
                'secret_key' => AppSetting::get('payment_secret'),
                'sandbox' => AppSetting::get('stripe_sandbox', true),
            ],
            'sslcommerz' => [
                'store_id' => AppSetting::get('sslcommerz_store_id'),
                'store_password' => AppSetting::get('sslcommerz_store_password'),
                'sandbox' => AppSetting::get('sslcommerz_sandbox', true),
            ],
            'bkash' => [
                'number' => AppSetting::get('bkash_number'),
                'type' => AppSetting::get('bkash_type', 'personal'),
                'enabled' => AppSetting::get('enable_bkash_manual', false),
            ],
            'nagad' => [
                'number' => AppSetting::get('nagad_number'),
                'type' => AppSetting::get('nagad_type', 'personal'),
                'enabled' => AppSetting::get('enable_nagad_manual', false),
            ],
            'rocket' => [
                'number' => AppSetting::get('rocket_number'),
                'type' => AppSetting::get('rocket_type', 'personal'),
                'enabled' => AppSetting::get('enable_rocket_manual', false),
            ],
        ];
    }

    public function getActiveGateways()
    {
        $gateways = [];
        
        // Check Stripe
        if ($this->config['stripe']['public_key'] && $this->config['stripe']['secret_key']) {
            $gateways['stripe'] = [
                'name' => 'Stripe',
                'type' => 'card',
                'enabled' => true,
                'sandbox' => $this->config['stripe']['sandbox']
            ];
        }

        // Check SSLCommerz
        if ($this->config['sslcommerz']['store_id']) {
            $gateways['sslcommerz'] = [
                'name' => 'SSLCommerz',
                'type' => 'gateway',
                'enabled' => true,
                'sandbox' => $this->config['sslcommerz']['sandbox']
            ];
        }

        // Check manual gateways
        if ($this->config['bkash']['enabled'] && $this->config['bkash']['number']) {
            $gateways['bkash'] = [
                'name' => 'bKash',
                'type' => 'manual',
                'enabled' => true,
                'number' => $this->config['bkash']['number']
            ];
        }

        if ($this->config['nagad']['enabled'] && $this->config['nagad']['number']) {
            $gateways['nagad'] = [
                'name' => 'Nagad',
                'type' => 'manual',
                'enabled' => true,
                'number' => $this->config['nagad']['number']
            ];
        }

        if ($this->config['rocket']['enabled'] && $this->config['rocket']['number']) {
            $gateways['rocket'] = [
                'name' => 'Rocket',
                'type' => 'manual',
                'enabled' => true,
                'number' => $this->config['rocket']['number']
            ];
        }

        return $gateways;
    }

    public function processPayment($gateway, $amount, $currency, $description, $metadata = [])
    {
        try {
            switch ($gateway) {
                case 'stripe':
                    return $this->processStripePayment($amount, $currency, $description, $metadata);
                case 'sslcommerz':
                    return $this->processSSLCommerzPayment($amount, $currency, $description, $metadata);
                case 'bkash':
                case 'nagad':
                case 'rocket':
                    return $this->processManualPayment($gateway, $amount, $currency, $description, $metadata);
                default:
                    throw new \Exception("Payment gateway '{$gateway}' is not supported");
            }
        } catch (\Exception $e) {
            Log::error("Payment processing failed for gateway {$gateway}: " . $e->getMessage());
            throw $e;
        }
    }

    protected function processStripePayment($amount, $currency, $description, $metadata)
    {
        Stripe::setApiKey($this->config['stripe']['secret_key']);

        // Create Payment Intent
        $intentData = [
            'amount' => $amount * 100, // Stripe uses cents
            'currency' => strtolower($currency),
            'description' => $description,
            'metadata' => $metadata,
            'payment_method_types' => ['card']
        ];

        $paymentIntent = PaymentIntent::create($intentData);

        return [
            'success' => true,
            'transaction_id' => $paymentIntent->id,
            'gateway' => 'stripe',
            'amount' => $amount,
            'currency' => $currency,
            'status' => 'requires_payment_method',
            'payment_method' => 'stripe_payment_intent',
            'client_secret' => $paymentIntent->client_secret,
            'created_at' => date('Y-m-d H:i:s', $paymentIntent->created)
        ];
    }

    protected function processSSLCommerzPayment($amount, $currency, $description, $metadata)
    {
        // SSLCommerz implementation would go here
        // For now, simulate the payment
        $transactionId = 'SSL_' . uniqid();
        
        return [
            'success' => true,
            'transaction_id' => $transactionId,
            'gateway' => 'sslcommerz',
            'amount' => $amount,
            'currency' => $currency,
            'status' => 'completed',
            'payment_method' => 'online',
            'redirect_url' => "https://sandbox.sslcommerz.com/gwprocess/v4/gateway.php?tran_id={$transactionId}",
            'created_at' => now()->toDateTimeString()
        ];
    }

    protected function processManualPayment($gateway, $amount, $currency, $description, $metadata)
    {
        $transactionId = strtoupper($gateway) . '_' . uniqid();
        
        return [
            'success' => true,
            'transaction_id' => $transactionId,
            'gateway' => $gateway,
            'amount' => $amount,
            'currency' => $currency,
            'status' => 'pending_manual_verification',
            'payment_method' => 'manual',
            'instructions' => $this->getManualPaymentInstructions($gateway),
            'created_at' => now()->toDateTimeString()
        ];
    }

    protected function getOrCreateStripeCustomer($tenantId, $tenantName)
    {
        // Try to find existing customer
        $existingCustomers = StripeCustomer::all(['email' => "tenant_{$tenantId}@example.com"]);
        
        if ($existingCustomers->count() > 0) {
            return $existingCustomers->first();
        }

        // Create new customer
        return StripeCustomer::create([
            'email' => "tenant_{$tenantId}@example.com",
            'name' => $tenantName,
            'metadata' => ['tenant_id' => $tenantId]
        ]);
    }

    protected function getManualPaymentInstructions($gateway)
    {
        $instructions = [
            'bkash' => [
                'title' => 'bKash Payment Instructions',
                'steps' => [
                    '1. Go to your bKash mobile app',
                    '2. Select "Send Money"',
                    '3. Enter the bKash number: ' . $this->config['bkash']['number'],
                    '4. Enter the amount and complete the payment',
                    '5. Take a screenshot of the successful transaction',
                    '6. Contact support with the transaction ID and screenshot'
                ]
            ],
            'nagad' => [
                'title' => 'Nagad Payment Instructions',
                'steps' => [
                    '1. Go to your Nagad mobile app',
                    '2. Select "Send Money"',
                    '3. Enter the Nagad number: ' . $this->config['nagad']['number'],
                    '4. Enter the amount and complete the payment',
                    '5. Take a screenshot of the successful transaction',
                    '6. Contact support with the transaction ID and screenshot'
                ]
            ],
            'rocket' => [
                'title' => 'Rocket Payment Instructions',
                'steps' => [
                    '1. Go to your Rocket mobile app',
                    '2. Select "Send Money"',
                    '3. Enter the Rocket number: ' . $this->config['rocket']['number'],
                    '4. Enter the amount and complete the payment',
                    '5. Take a screenshot of the successful transaction',
                    '6. Contact support with the transaction ID and screenshot'
                ]
            ]
        ];

        return $instructions[$gateway] ?? [];
    }

    public function verifyPayment($gateway, $transactionId)
    {
        // This would verify the payment status with the gateway
        // For now, return success for demonstration
        return [
            'success' => true,
            'status' => 'completed',
            'verified_at' => now()->toDateTimeString()
        ];
    }

    public function refundPayment($gateway, $transactionId, $amount = null)
    {
        try {
            switch ($gateway) {
                case 'stripe':
                    return $this->refundStripePayment($transactionId, $amount);
                default:
                    throw new \Exception("Refunds not supported for gateway '{$gateway}'");
            }
        } catch (\Exception $e) {
            Log::error("Refund failed for gateway {$gateway}: " . $e->getMessage());
            throw $e;
        }
    }

    protected function refundStripePayment($transactionId, $amount)
    {
        Stripe::setApiKey($this->config['stripe']['secret_key']);
        
        $charge = Charge::retrieve($transactionId);
        $refund = $charge->refund(['amount' => $amount ? $amount * 100 : null]);

        return [
            'success' => true,
            'refund_id' => $refund->id,
            'amount' => $refund->amount / 100,
            'status' => $refund->status,
            'created_at' => date('Y-m-d H:i:s', $refund->created)
        ];
    }
}
