<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class AppSetting extends Model
{
    protected $fillable = ['key', 'value', 'group', 'type'];

    /**
     * Get the table associated with the model.
     */
    public function getTable()
    {
        return 'app_settings';
    }

    /**
     * Get a setting by key.
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function get($key, $default = null)
    {
        // Check if we're in a tenant context
        $isTenant = request()->segment(1) === 'tenant';
        
        if ($isTenant) {
            $setting = DB::connection('tenant')->table('app_settings')->where('key', $key)->first();
        } else {
            $setting = self::where('key', $key)->first();
        }
        
        return $setting ? $setting->value : $default;
    }

    /**
     * Set a setting.
     *
     * @param string $key
     * @param mixed $value
     * @param string $group
     * @param string $type
     * @return void
     */
    public static function set($key, $value, $group = 'general', $type = 'string')
    {
        // Check if we're in a tenant context
        $isTenant = request()->segment(1) === 'tenant';
        
        if ($isTenant) {
            DB::connection('tenant')->table('app_settings')->updateOrInsert(
                ['key' => $key],
                [
                    'value' => $value,
                    'group' => $group,
                    'type' => $type,
                    'updated_at' => now()
                ]
            );
        } else {
            self::updateOrCreate(
                ['key' => $key],
                ['value' => $value, 'group' => $group, 'type' => $type]
            );
        }
    }
}
