<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use App\Models\Tenant;
use App\Services\TenantService;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;

class TenantResolver
{
    protected $tenantService;

    public function __construct(TenantService $tenantService)
    {
        $this->tenantService = $tenantService;
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $tenantId = $request->header('X-Tenant-ID');
        $domain = $request->getHost();

        $tenant = null;

        if ($tenantId) {
            $tenant = Tenant::where('id', $tenantId)->orWhere('domain', $tenantId)->first();
        } 
        
        // If not found by ID, try domain (basic check, usually requires subdomain parsing)
        if (!$tenant && $domain) {
             $tenant = Tenant::where('domain', $domain)->first();
        }

        if ($tenant && $tenant->status === 'active') {
            // Set the tenant in the service/container
            $this->tenantService->switchToTenant($tenant);
            
            // Store tenant in request for easy access
            $request->merge(['tenant' => $tenant]);
            
            // Set default connection to tenant for this request scope?
            // Depends on strategy. Let's set it to 'tenant' to simplify model usage.
            DB::setDefaultConnection('tenant');
        } else {
            // If strictly tenant-scoped route, abort.
            // But some routes are central. We need to know if this route requires tenancy.
            // For now, if no tenant is found, we stay on central (mysql).
            // However, if the user explicitly requested a tenant and it's invalid, maybe 404?
            if ($tenantId) {
                return response()->json(['message' => 'Tenant not found or inactive'], 404);
            }
        }

        return $next($request);
    }
}
