<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use App\Models\Tenant\Setting;
use Illuminate\Support\Facades\Storage;

class SetupTenantContext
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = Auth::user();

        if ($user && $user->tenant) {
            $tenant = $user->tenant;
            
            if ($tenant->status !== 'active') {
                Auth::logout();
                return redirect()->route('login')->withErrors(['email' => 'Your tenant account is not active.']);
            }

            // Switch Database Connection
            Config::set('database.connections.tenant.database', $tenant->database_name);
            DB::purge('tenant');
            DB::reconnect('tenant');
            DB::setDefaultConnection('tenant');

            // Apply Tenant Settings
            try {
                $timezone = Setting::get('timezone', 'UTC');
                $dateFormat = Setting::get('date_format', 'Y-m-d');
                $storeName = Setting::get('store_name', $tenant->name);
                $themeColor = Setting::get('theme_color', '#10B981');
                
                // Files
                $logoPath = Setting::get('tenant_logo');
                $iconPath = Setting::get('tenant_icon');
                $logoUrl = $logoPath ? Storage::url($logoPath) : null;
                $iconUrl = $iconPath ? Storage::url($iconPath) : null;

                // Apply Runtime Config
                Config::set('app.timezone', $timezone);
                date_default_timezone_set($timezone);
                
                // Share settings with all views
                view()->share('tenantSettings', [
                    'store_name' => $storeName,
                    'date_format' => $dateFormat,
                    'theme_color' => $themeColor,
                    'logo_url' => $logoUrl,
                    'icon_url' => $iconUrl,
                    'currency' => Setting::get('currency', 'USD'),
                    'currency_symbol' => Setting::get('currency_symbol', '$'),
                ]);

            } catch (\Exception $e) {
                // Fallback if settings table doesn't exist yet (e.g. during migration)
                // or other DB issues
            }

            // Share tenant with all views
            view()->share('tenant', $tenant);
        }

        return $next($request);
    }
}
