<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Services\TenantService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use App\Models\SecuritySetting;
use Illuminate\Support\Facades\Mail;
use App\Mail\UserCredentialsMail;

class WebAuthController extends Controller
{
    protected $tenantService;

    public function __construct(TenantService $tenantService)
    {
        $this->tenantService = $tenantService;
    }

    public function showRegisterForm()
    {
        $categories = TenantCategory::where('is_active', true)->get();
        return view('auth.register', compact('categories'));
    }

    public function register(Request $request)
    {
        $validated = $request->validate([
            'business_name' => 'required|string|max:255',
            'tenant_category_id' => 'required|exists:tenant_categories,id',
            'domain' => 'required|string|unique:tenants,domain|max:255', // subdomain
            'name' => 'required|string|max:255', // User name
            'email' => 'required|email|unique:users,email',
            'phone_number' => 'required|string|unique:users,phone_number|regex:/^01[3-9][0-9]{8}$/',
        ]);

        try {
            DB::beginTransaction();

            // 1. Prepare Tenant Data
            $fullDomain = $validated['domain'] . '.grocery-saas.local';
            $dbName = 'tenant_' . str_replace('-', '_', \Illuminate\Support\Str::slug($fullDomain));
            
            $tenantData = [
                'name' => $validated['business_name'],
                'tenant_category_id' => $validated['tenant_category_id'],
                'domain' => $fullDomain,
                'database_name' => $dbName,
                'status' => 'active', 
            ];

            // 2. Prepare User Data (Manager)
            $generatedPassword = \Illuminate\Support\Str::random(12);
            $userData = [
                'name' => $validated['name'],
                'email' => $validated['email'],
                'phone_number' => $validated['phone_number'],
                'password' => $generatedPassword,
            ];

            // 3. Create Tenant and Users via Service
            $tenant = $this->tenantService->createTenant($tenantData, $userData);

            // 4. Send Credentials to Manager via Email
            $user = User::where('email', $validated['email'])->first();
            try {
                Mail::to($user->email)->send(new UserCredentialsMail($user, $generatedPassword, true));
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error('Failed to send registration email: ' . $e->getMessage());
            }

            DB::commit();

            return redirect()->route('tenant.login')->with('success', "Registration successful! Your store '{$validated['business_name']}' has been created at {$fullDomain}. We have sent the manager password to your email: {$user->email}");

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Registration failed: ' . $e->getMessage()])->withInput();
        }
    }

    public function showLoginForm()
    {
        $security = SecuritySetting::first();
        return view('auth.login', compact('security'));
    }

    public function login(Request $request)
    {
        $security = SecuritySetting::first();

        // 1. Captcha Validation
        if ($security && $security->enable_captcha) {
            $request->validate([
                'captcha' => 'required|captcha'
            ], [
                'captcha.required' => 'Please verify that you are not a robot.',
                'captcha.captcha' => 'Invalid captcha code. Please try again.'
            ]);
        }

        $credentials = $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);

        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();

            // Security Check: Allow SaaS Admin (Role: Admin) and Developers (Role: Developer) to login here
            // If the user does not have the 'Admin' or 'Developer' role, logout and deny access
            if (!Auth::user()->hasRole('Admin') && !Auth::user()->hasRole('Developer')) {
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                
                return back()->withErrors([
                    'email' => 'Access denied. This login is restricted to SaaS Administrators and Developers only.',
                ]);
            }

            // Redirect based on user role
            if (Auth::user()->hasRole('Developer')) {
                return redirect()->intended('developer.dashboard');
            } else {
                return redirect()->intended('dashboard');
            }
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->onlyInput('email');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }
}
