<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\Unit;

class UnitController extends Controller
{
    public function index()
    {
        // Fetch base units along with their sub-units for better display
        $baseUnits = Unit::on('tenant')->whereNull('base_unit_id')->with('subUnits')->orderBy('name')->get();
        return view('tenant.catalog.units.index', compact('baseUnits'));
    }

    public function create()
    {
        $baseUnits = Unit::on('tenant')->whereNull('base_unit_id')->get();
        return view('tenant.catalog.units.create', compact('baseUnits'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_name' => 'required|string|max:50',
            // Specify connection for exists rule
            'base_unit_id' => 'nullable|exists:tenant.units,id',
            'conversion_factor' => 'nullable|numeric|min:0',
            'type' => 'required|in:base,purchase,selling',
        ]);

        // If it's a base unit, conversion factor is always 1
        if (empty($validated['base_unit_id'])) {
            $validated['conversion_factor'] = 1;
            $validated['type'] = 'base';
        } else {
            // Validate conversion factor if sub-unit
            if (empty($validated['conversion_factor']) || $validated['conversion_factor'] <= 0) {
                return back()->withErrors(['conversion_factor' => 'Conversion factor is required for sub-units.'])->withInput();
            }
        }

        Unit::create($validated);
        return redirect()->route('tenant.units.index')->with('success', 'Unit created successfully.');
    }

    public function edit($id)
    {
        $unit = Unit::on('tenant')->findOrFail($id);
        $baseUnits = Unit::on('tenant')->whereNull('base_unit_id')->where('id', '!=', $unit->id)->get();
        return view('tenant.catalog.units.edit', compact('unit', 'baseUnits'));
    }

    public function update(Request $request, $id)
    {
        $unit = Unit::on('tenant')->findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_name' => 'required|string|max:50',
            // Specify connection for exists rule
            'base_unit_id' => 'nullable|exists:tenant.units,id',
            'conversion_factor' => 'nullable|numeric|min:0',
            'type' => 'required|in:base,purchase,selling',
        ]);

        if (empty($validated['base_unit_id'])) {
            $validated['conversion_factor'] = 1;
            $validated['type'] = 'base';
        }

        $unit->update($validated);
        return redirect()->route('tenant.units.index')->with('success', 'Unit updated successfully.');
    }

    public function destroy($id)
    {
        $unit = Unit::on('tenant')->findOrFail($id);
        
        if ($unit->subUnits()->exists()) {
            return back()->with('error', 'Cannot delete base unit with existing sub-units.');
        }
        
        // Check if used in products (optional, but recommended)
        // if ($unit->products()->exists()) { ... }

        $unit->delete();
        return redirect()->route('tenant.units.index')->with('success', 'Unit deleted successfully.');
    }
}
