<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Plan;
use App\Models\Subscription;
use App\Services\PaymentService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class SubscriptionController extends Controller
{
    protected $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    public function index()
    {
        $plans = Plan::where('is_active', true)->get();
        $tenant = Auth::user()->tenant;
        $gateways = $this->paymentService->getActiveGateways();
        
        return view('tenant.subscription.index', compact('plans', 'tenant', 'gateways'));
    }

    public function store(Request $request, Plan $plan)
    {
        $request->validate([
            'payment_gateway' => 'required|in:stripe,sslcommerz,bkash,nagad,rocket',
        ]);

        $tenant = Auth::user()->tenant;
        
        if (!$tenant) {
            abort(403, 'Tenant not found');
        }

        // Check if tenant already has an active subscription
        $existingSubscription = $tenant->subscription()
            ->where('status', 'active')
            ->where('ends_at', '>', now())
            ->first();

        if ($existingSubscription) {
            return back()->with('error', 'You already have an active subscription.');
        }

        try {
            // Process payment
            $result = $this->paymentService->processPayment(
                $request->payment_gateway,
                $plan->price,
                'USD',
                "Subscription to {$plan->name} for {$tenant->name}",
                [
                    'tenant_id' => $tenant->id,
                    'tenant_name' => $tenant->name,
                    'plan_id' => $plan->id,
                    'plan_name' => $plan->name,
                    'user_id' => Auth::id(),
                    'user_email' => Auth::user()->email
                ]
            );

            if ($result['success']) {
                // Create subscription record
                $ends_at = match($plan->interval) {
                    'daily' => now()->addDays(1),
                    'monthly' => now()->addDays(30), // 30 Day = 1 Month
                    'yearly' => now()->addDays(360), // 12 * 30 Days = 360 Days or 12 Months
                    default => now()->addMonths(1),
                };

                $subscription = $tenant->subscription()->create([
                    'plan_id' => $plan->id,
                    'status' => 'active',
                    'starts_at' => now(),
                    'ends_at' => $ends_at,
                    'trial_ends_at' => null,
                    'payment_gateway' => $result['gateway'],
                    'transaction_id' => $result['transaction_id'],
                    'amount' => $result['amount'],
                    'currency' => $result['currency'],
                    'payment_status' => 'paid'
                ]);

                Log::info('Subscription purchased successfully', [
                    'tenant_id' => $tenant->id,
                    'plan_id' => $plan->id,
                    'transaction_id' => $result['transaction_id'],
                    'amount' => $result['amount']
                ]);

                // For manual payments, show instructions
                if ($result['status'] === 'pending_manual_verification') {
                    return view('tenant.subscription.payment-instructions', [
                        'subscription' => $subscription,
                        'payment' => $result
                    ]);
                }

                // For online payments, redirect to success page
                if (isset($result['redirect_url'])) {
                    return redirect($result['redirect_url']);
                }

                return redirect()->route('tenant.dashboard')
                    ->with('success', "Subscription to {$plan->name} purchased successfully!");
            } else {
                return back()->with('error', 'Payment failed. Please try again.');
            }

        } catch (\Exception $e) {
            Log::error('Payment processing failed', [
                'tenant_id' => $tenant->id,
                'plan_id' => $plan->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('error', 'Payment processing failed: ' . $e->getMessage());
        }
    }

    public function paymentSuccess(Request $request)
    {
        $transactionId = $request->input('transaction_id');
        $gateway = $request->input('gateway');

        // Verify payment
        $verification = $this->paymentService->verifyPayment($gateway, $transactionId);

        if ($verification['success']) {
            // Update subscription status if needed
            $subscription = Subscription::where('transaction_id', $transactionId)->first();
            if ($subscription) {
                $subscription->update([
                    'payment_status' => 'verified',
                    'verified_at' => now()
                ]);
            }

            return redirect()->route('tenant.dashboard')
                ->with('success', 'Payment verified successfully!');
        } else {
            return redirect()->route('tenant.subscription.index')
                ->with('error', 'Payment verification failed. Please contact support.');
        }
    }

    public function paymentCallback(Request $request)
    {
        // Handle payment gateway callbacks (webhooks)
        $gateway = $request->input('gateway');
        $transactionId = $request->input('transaction_id');
        $status = $request->input('status');

        Log::info('Payment callback received', [
            'gateway' => $gateway,
            'transaction_id' => $transactionId,
            'status' => $status,
            'data' => $request->all()
        ]);

        // Update subscription based on callback
        $subscription = Subscription::where('transaction_id', $transactionId)->first();
        if ($subscription) {
            $subscription->update([
                'payment_status' => $status === 'success' ? 'paid' : 'failed',
                'status' => $status === 'success' ? 'active' : 'cancelled'
            ]);
        }

        return response()->json(['status' => 'received']);
    }
}
