<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\Sale;
use App\Models\Tenant\SaleItem;
use App\Models\Tenant\Product;
use App\Models\Tenant\Purchase;
use App\Models\Tenant\Supplier;
use App\Models\Tenant\Customer;
use App\Models\Tenant\ProductVariant;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;

class ReportController extends Controller
{
    public function index()
    {
        // Overview Stats for Cards
        $todaySales = Sale::whereDate('created_at', Carbon::today())->sum('total');
        $monthSales = Sale::whereMonth('created_at', Carbon::now()->month)->sum('total');
        $totalOrders = Sale::count();
        $averageOrderValue = $totalOrders > 0 ? Sale::sum('total') / $totalOrders : 0;

        return view('tenant.reports.index', compact('todaySales', 'monthSales', 'totalOrders', 'averageOrderValue'));
    }

    public function detailed(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        
        $customerId = $request->input('customer_id');
        $supplierId = $request->input('supplier_id');
        $variantId = $request->input('variant_id');
        $categoryId = $request->input('category_id');
        $subcategoryId = $request->input('subcategory_id');
        $productId = $request->input('product_id');
        
        $reportType = $request->input('type', 'sales'); // 'sales', 'purchases', or 'credit'

        $results = [];

        $selectedProduct = null;
        if ($productId) {
            $selectedProduct = Product::find($productId);
        }

        $selectedSubcategory = null;
        if ($subcategoryId) {
            $selectedSubcategory = \App\Models\Tenant\Category::find($subcategoryId);
        }

        if ($reportType === 'sales') {
            $query = Sale::with(['customer', 'items.variant.product'])
                ->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);

            if ($customerId) {
                $query->where('customer_id', $customerId);
            }

            if ($variantId) {
                $query->whereHas('items', function($q) use ($variantId) {
                    $q->where('variant_id', $variantId);
                });
            }

            if ($categoryId) {
                $query->whereHas('items.variant.product', function($q) use ($categoryId) {
                    $q->where('category_id', $categoryId);
                });
            }

            if ($subcategoryId) {
                $query->whereHas('items.variant.product', function($q) use ($subcategoryId) {
                    $q->where('subcategory_id', $subcategoryId);
                });
            }

            if ($productId) {
                $query->whereHas('items.variant.product', function($q) use ($productId) {
                    $q->where('id', $productId);
                });
            }

            $results = $query->latest()->get();
        } elseif ($reportType === 'purchases') {
            $query = Purchase::with(['supplier', 'items.variant.product'])
                ->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);

            if ($supplierId) {
                $query->where('supplier_id', $supplierId);
            }

            if ($variantId) {
                $query->whereHas('items', function($q) use ($variantId) {
                    $q->where('product_variant_id', $variantId);
                });
            }

            if ($categoryId) {
                $query->whereHas('items.variant.product', function($q) use ($categoryId) {
                    $q->where('category_id', $categoryId);
                });
            }

            if ($subcategoryId) {
                $query->whereHas('items.variant.product', function($q) use ($subcategoryId) {
                    $q->where('subcategory_id', $subcategoryId);
                });
            }

            if ($productId) {
                $query->whereHas('items.variant.product', function($q) use ($productId) {
                    $q->where('id', $productId);
                });
            }

            $results = $query->latest()->get();
        } elseif ($reportType === 'credit') {
            // Credit/Due Report
            $query = Sale::with(['customer', 'items.variant.product'])
                ->whereIn('payment_status', ['due', 'partial'])
                ->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);

            if ($customerId) {
                $query->where('customer_id', $customerId);
            }

            if ($categoryId) {
                $query->whereHas('items.variant.product', function($q) use ($categoryId) {
                    $q->where('category_id', $categoryId);
                });
            }

            if ($subcategoryId) {
                $query->whereHas('items.variant.product', function($q) use ($subcategoryId) {
                    $q->where('subcategory_id', $subcategoryId);
                });
            }

            if ($productId) {
                $query->whereHas('items.variant.product', function($q) use ($productId) {
                    $q->where('id', $productId);
                });
            }

            $results = $query->latest()->get();
        } elseif ($reportType === 'returns') {
            // Returns Report
            $query = \App\Models\Tenant\SaleReturn::with(['customer', 'sale', 'items.variant.product'])
                ->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);

            if ($customerId) {
                $query->where('customer_id', $customerId);
            }

            if ($categoryId) {
                $query->whereHas('items.variant.product', function($q) use ($categoryId) {
                    $q->where('category_id', $categoryId);
                });
            }

            if ($subcategoryId) {
                $query->whereHas('items.variant.product', function($q) use ($subcategoryId) {
                    $q->where('subcategory_id', $subcategoryId);
                });
            }

            if ($productId) {
                $query->whereHas('items.variant.product', function($q) use ($productId) {
                    $q->where('id', $productId);
                });
            }

            $results = $query->latest()->get();
        } elseif ($reportType === 'expiring_products') {
            // Expiring Products Report
            $query = Product::with('variants')
                ->whereNotNull('expiry_date')
                ->where('expiry_date', '<=', Carbon::now()->addDays(7));

            if ($categoryId) {
                $query->where('category_id', $categoryId);
            }

            if ($subcategoryId) {
                $query->where('subcategory_id', $subcategoryId);
            }

            if ($productId) {
                $query->where('id', $productId);
            }

            $results = $query->latest('expiry_date')->get();
        }

        // Data for filters
        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();
        $products = ProductVariant::with('product')->get()->map(function($v) {
            $v->full_name = $v->product->name . ($v->name !== $v->product->name ? ' - ' . $v->name : '');
            return $v;
        })->sortBy('full_name');
        $categories = \App\Models\Tenant\Category::orderBy('name')->pluck('name', 'id');

        if ($reportType === 'expiring_products') {
            return view('tenant.reports.expiring-products', compact('results', 'categories', 'categoryId', 'subcategoryId', 'productId', 'selectedProduct'));
        }

        return view('tenant.reports.detailed', compact(
            'results', 'reportType', 'startDate', 'endDate', 
            'customers', 'suppliers', 'products', 
            'customerId', 'supplierId', 'variantId',
            'categories', 'categoryId', 'subcategoryId', 'productId', 'selectedProduct', 'selectedSubcategory'
        ));
    }

    public function sales(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));

        $salesData = Sale::select(
                DB::raw('DATE(created_at) as date'), 
                DB::raw('SUM(total) as total_sales'),
                DB::raw('COUNT(*) as order_count')
            )
            ->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59'])
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return view('tenant.reports.sales', compact('salesData', 'startDate', 'endDate'));
    }

    public function topProducts(Request $request)
    {
        $limit = $request->input('limit', 10);

        $topProducts = SaleItem::select(
                'product_variants.name as variant_name',
                'products.name as product_name',
                DB::raw('SUM(sale_items.quantity) as total_quantity'),
                DB::raw('SUM(sale_items.total) as total_revenue')
            )
            ->join('product_variants', 'sale_items.variant_id', '=', 'product_variants.id')
            ->join('products', 'product_variants.product_id', '=', 'products.id')
            ->groupBy('product_variants.id', 'product_variants.name', 'products.name')
            ->orderByDesc('total_quantity')
            ->limit($limit)
            ->get();

        return view('tenant.reports.top-products', compact('topProducts'));
    }

    public function inventory()
    {
        // Low stock report
        $lowStockProducts = Product::with('variants')
            ->whereHas('variants', function($q) {
                $q->where('stock', '<', 10); // Threshold 10
            })
            ->get()
            ->map(function($product) {
                return $product->variants->where('stock', '<', 10)->map(function($variant) use ($product) {
                    return [
                        'product_name' => $product->name,
                        'variant_name' => $variant->name,
                        'sku' => $variant->sku,
                        'stock' => $variant->stock,
                    ];
                });
            })
            ->flatten(1);

        return view('tenant.reports.inventory', compact('lowStockProducts'));
    }

    public function stockReport(Request $request)
    {
        $categoryId = $request->input('category_id');
        $subcategoryId = $request->input('subcategory_id');
        $stockStatus = $request->input('stock_status', 'all'); // all, low, out, normal
        $search = $request->input('search');
        $productId = $request->input('product_id');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $selectedProduct = null;
        if ($productId) {
            $selectedProduct = Product::find($productId);
        }

        $query = Product::with(['variants', 'category', 'unit'])
            ->select('products.*', 'categories.name as category_name', 'units.name as unit_name')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->join('units', 'products.unit_id', '=', 'units.id');

        if ($categoryId) {
            $query->where('products.category_id', $categoryId);
        }

        if ($subcategoryId) {
            $query->where('products.subcategory_id', $subcategoryId);
        }

        if ($productId) {
            $query->where('products.id', $productId);
        }

        if ($startDate) {
            $query->whereDate('products.created_at', '>=', $startDate);
        }

        if ($endDate) {
            $query->whereDate('products.created_at', '<=', $endDate);
        }

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('products.name', 'like', "%{$search}%")
                  ->orWhereHas('variants', function($vq) use ($search) {
                      $vq->where('name', 'like', "%{$search}%")
                         ->orWhere('sku', 'like', "%{$search}%")
                         ->orWhere('barcode', 'like', "%{$search}%");
                  });
            });
        }

        $products = $query->get()->map(function($product) use ($stockStatus) {
            $variants = $product->variants->map(function($variant) {
                $stockValue = $variant->stock * $variant->price;
                $status = $this->getStockStatus($variant->stock);
                
                return [
                    'id' => $variant->id,
                    'name' => $variant->name,
                    'sku' => $variant->sku,
                    'barcode' => $variant->barcode,
                    'stock' => $variant->stock,
                    'price' => $variant->price,
                    'stock_value' => $stockValue,
                    'status' => $status,
                    'status_color' => $this->getStockStatusColor($status),
                ];
            });

            // Filter by stock status if specified
            if ($stockStatus !== 'all') {
                $variants = $variants->filter(function($variant) use ($stockStatus) {
                    return $variant['status'] === $stockStatus;
                });
            }

            return [
                'id' => $product->id,
                'name' => $product->name,
                'category' => $product->category_name,
                'unit' => $product->unit_name,
                'variants' => $variants,
                'total_stock' => $variants->sum('stock'),
                'total_value' => $variants->sum('stock_value'),
            ];
        });

        // Calculate summary statistics
        $totalProducts = $products->count();
        $totalVariants = $products->sum(function($product) {
            return $product['variants']->count();
        });
        $totalStockValue = $products->sum('total_value');
        $lowStockCount = $products->sum(function($product) {
            return $product['variants']->filter(function($variant) {
                return $variant['status'] === 'low';
            })->count();
        });
        $outOfStockCount = $products->sum(function($product) {
            return $product['variants']->filter(function($variant) {
                return $variant['status'] === 'out';
            })->count();
        });

        // Get categories for filter
        $categories = \App\Models\Tenant\Category::orderBy('name')->pluck('name', 'id');

        return view('tenant.reports.stock', compact(
            'products', 'categories', 'categoryId', 'stockStatus', 'search', 'selectedProduct',
            'totalProducts', 'totalVariants', 'totalStockValue', 
            'lowStockCount', 'outOfStockCount', 'startDate', 'endDate'
        ));
    }

    public function stockMovement(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->format('Y-m-d'));
        $variantId = $request->input('variant_id');
        $movementType = $request->input('movement_type', 'all'); // all, sale, purchase, adjustment

        $query = DB::connection('tenant')
            ->table('stock_movements')
            ->select(
                'stock_movements.*',
                'product_variants.name as variant_name',
                'products.name as product_name'
            )
            ->join('product_variants', 'stock_movements.variant_id', '=', 'product_variants.id')
            ->join('products', 'product_variants.product_id', '=', 'products.id')
            ->whereBetween('stock_movements.created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);

        if ($variantId) {
            $query->where('stock_movements.variant_id', $variantId);
        }

        if ($movementType !== 'all') {
            $query->where('stock_movements.movement_type', $movementType);
        }

        $movements = $query->latest()->get()->map(function($movement) {
            return [
                'id' => $movement->id,
                'date' => Carbon::parse($movement->created_at)->format('Y-m-d H:i:s'),
                'product_name' => $movement->product_name,
                'variant_name' => $movement->variant_name,
                'movement_type' => $movement->movement_type,
                'quantity' => $movement->quantity,
                'reference_type' => $movement->reference_type,
                'reference_id' => $movement->reference_id,
                'notes' => $movement->notes,
            ];
        });

        // Get variants for filter
        $variants = ProductVariant::with('product')->get()->map(function($v) {
            $v->full_name = $v->product->name . ' - ' . $v->name;
            return $v;
        })->sortBy('full_name');

        return view('tenant.reports.stock-movement', compact(
            'movements', 'variants', 'variantId', 'movementType',
            'startDate', 'endDate'
        ));
    }

    private function getStockStatus($stock)
    {
        if ($stock <= 0) {
            return 'out';
        } elseif ($stock < 10) {
            return 'low';
        } else {
            return 'normal';
        }
    }

    private function getStockStatusColor($status)
    {
        return match($status) {
            'out' => 'danger',
            'low' => 'warning',
            'normal' => 'success',
            default => 'secondary'
        };
    }

    public function exportStockReport(Request $request)
    {
        $categoryId = $request->input('category_id');
        $subcategoryId = $request->input('subcategory_id');
        $stockStatus = $request->input('stock_status', 'all');
        $search = $request->input('search');
        $productId = $request->input('product_id');
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        return Excel::download(new \App\Exports\StockReportExport(
            $categoryId, $subcategoryId, $stockStatus, $search, $productId, $startDate, $endDate
        ), 'stock_report_' . Carbon::now()->format('Ymd_His') . '.xlsx');
    }
}
