<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\PurchaseRequisition;
use App\Models\Tenant\ProductVariant;
use App\Models\Tenant\Supplier;
use App\Models\Tenant\Category;
use App\Models\Tenant\Brand;
use App\Models\Tenant\Unit;
use App\Models\Tenant\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class PurchaseRequisitionController extends Controller
{
    public function index()
    {
        $requisitions = PurchaseRequisition::with('supplier')->latest()->paginate(10);
        return view('tenant.inventory.requisitions.index', compact('requisitions'));
    }

    public function create()
    {
        $suppliers = Supplier::all();
        // Load low stock products (e.g., stock <= 10)
        $lowStockProducts = ProductVariant::where('stock', '<=', 10)->with('product')->get();
        
        // Load all products for search
        $products = ProductVariant::with(['product.unit', 'product.brand', 'product.category'])
            ->get()
            ->filter(function($variant) {
                return $variant->product != null;
            })
            ->map(function($variant) {
                return [
                    'id' => $variant->id,
                    'name' => $variant->product->name . ($variant->name !== $variant->product->name ? ' - ' . $variant->name : ''),
                    'sku' => $variant->sku,
                    'stock' => $variant->stock,
                    'cost' => $variant->cost,
                    'price' => $variant->price,
                ];
            })->values();

        // Data for new product modal
        $categories = Category::all();
        $brands = Brand::all();
        $units = Unit::all();

        return view('tenant.inventory.requisitions.create', compact('suppliers', 'lowStockProducts', 'categories', 'brands', 'units', 'products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'nullable|exists:tenant.suppliers,id', // Nullable for "No Supplier"
            'purpose' => 'nullable|string|in:Low Stock,Low Price,Advance Stock,Other',
            'items' => 'required|array',
            'items.*.variant_id' => 'required|exists:tenant.product_variants,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.cost_price' => 'required|numeric|min:0',
            'note' => 'nullable|string',
        ]);

        try {
            DB::transaction(function () use ($validated) {
                $totalAmount = 0;
                foreach ($validated['items'] as $item) {
                    $totalAmount += $item['quantity'] * $item['cost_price'];
                }

                $requisition = PurchaseRequisition::create([
                    'requisition_no' => 'PR-' . strtoupper(uniqid()),
                    'supplier_id' => $validated['supplier_id'],
                    'purpose' => $validated['purpose'],
                    'total_amount' => $totalAmount,
                    'status' => 'pending',
                    'note' => $validated['note'],
                    'created_by' => Auth::id(),
                ]);

                foreach ($validated['items'] as $item) {
                    $requisition->items()->create([
                        'product_variant_id' => $item['variant_id'],
                        'quantity' => $item['quantity'],
                        'cost_price' => $item['cost_price'],
                        'total' => $item['quantity'] * $item['cost_price'],
                    ]);
                }
            });

            return redirect()->route('tenant.requisitions.index')->with('success', 'Requisition created successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to create requisition: ' . $e->getMessage());
        }
    }

    // Quick Store Methods for Modals
    public function storeSupplier(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'contact' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
        ]);
        
        $supplier = Supplier::create($validated);
        return response()->json(['success' => true, 'supplier' => $supplier]);
    }

    public function storeProduct(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'cost' => 'nullable|numeric|min:0',
            'stock' => 'nullable|integer|min:0',
            'category_id' => 'nullable|exists:tenant.categories,id',
            'brand_id' => 'nullable|exists:tenant.brands,id',
            'unit_id' => 'nullable|exists:tenant.units,id',
        ]);

        try {
            $product = null;
            $variant = null;

            DB::transaction(function () use ($validated, &$product, &$variant) {
                // If unit_id is missing, find or create a default 'piece' unit
                if (empty($validated['unit_id'])) {
                    $unit = Unit::firstOrCreate(
                        ['name' => 'Piece'],
                        ['short_name' => 'pc', 'type' => 'base', 'conversion_factor' => 1]
                    );
                    $validated['unit_id'] = $unit->id;
                }

                $product = Product::create([
                    'name' => $validated['name'],
                    'category_id' => $validated['category_id'] ?? null,
                    'brand_id' => $validated['brand_id'] ?? null,
                    'unit_id' => $validated['unit_id'],
                ]);

                $variant = $product->variants()->create([
                    'name' => $validated['name'],
                    'sku' => $validated['sku'],
                    'price' => $validated['price'],
                    'cost' => $validated['cost'] ?? 0,
                    'stock' => $validated['stock'] ?? 0,
                ]);
            });

            return response()->json([
                'success' => true, 
                'product' => [
                    'id' => $product->id, 
                    'name' => $product->name,
                    'variant_id' => $variant->id,
                    'cost' => $variant->cost
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        $requisition = PurchaseRequisition::on('tenant')->findOrFail($id);
        $requisition->load(['items.variant.product', 'supplier']);
        return view('tenant.inventory.requisitions.show', compact('requisition'));
    }

    public function approve($id)
    {
        try {
            $requisition = PurchaseRequisition::on('tenant')->findOrFail($id);
            
            if ($requisition->status !== 'pending') {
                return back()->with('error', 'Only pending requisitions can be approved.');
            }

            $requisition->update(['status' => 'approved']);

            return redirect()->route('tenant.requisitions.index')->with('success', 'Requisition approved successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to approve requisition: ' . $e->getMessage());
        }
    }
}
