<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\Purchase;
use App\Models\Tenant\PurchaseRequisition;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    public function index()
    {
        $purchases = Purchase::with('supplier')->latest()->paginate(10);
        return view('tenant.inventory.purchases.index', compact('purchases'));
    }

    public function createFromRequisition($id)
    {
        $requisition = PurchaseRequisition::findOrFail($id);
        
        if ($requisition->status !== 'pending') {
            return back()->with('error', 'Requisition is already processed.');
        }
        $requisition->load(['items.variant.product', 'supplier']);
        return view('tenant.inventory.purchases.create_from_pr', compact('requisition'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'purchase_requisition_id' => 'required|exists:tenant.purchase_requisitions,id',
            'supplier_id' => 'required|exists:tenant.suppliers,id',
            'items' => 'required|array',
            'items.*.variant_id' => 'required|exists:tenant.product_variants,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_cost' => 'required|numeric|min:0',
            'paid_amount' => 'required|numeric|min:0',
        ]);

        try {
            DB::connection('tenant')->transaction(function () use ($validated) {
                $requisition = PurchaseRequisition::findOrFail($validated['purchase_requisition_id']);
                
                // Update PR Status
                $requisition->update(['status' => 'received']); // Or 'ordered' if multi-step

                $totalAmount = 0;
                foreach ($validated['items'] as $item) {
                    $totalAmount += $item['quantity'] * $item['unit_cost'];
                }

                // Create Purchase
                $purchase = Purchase::create([
                    'purchase_no' => 'PO-' . strtoupper(uniqid()),
                    'purchase_requisition_id' => $requisition->id,
                    'supplier_id' => $validated['supplier_id'],
                    'total_amount' => $totalAmount,
                    'paid_amount' => $validated['paid_amount'],
                    'status' => 'received',
                ]);

                // Create Items and Update Stock
                foreach ($validated['items'] as $item) {
                    $purchase->items()->create([
                        'product_variant_id' => $item['variant_id'],
                        'quantity' => $item['quantity'],
                        'unit_cost' => $item['unit_cost'],
                        'total' => $item['quantity'] * $item['unit_cost'],
                    ]);

                    // Increase Stock
                    $variant = \App\Models\Tenant\ProductVariant::find($item['variant_id']);
                    $variant->increment('stock', $item['quantity']);
                    
                    // Update cost price if needed (Weighted Average could be better, but simple update for now)
                    $variant->update(['cost' => $item['unit_cost']]);
                }
            });

            return redirect()->route('tenant.purchases.index')->with('success', 'Purchase processed and stock updated successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to process purchase: ' . $e->getMessage());
        }
    }
}
