<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\AdminNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class NotificationController extends Controller
{
    public function index()
    {
        $tenant = Auth::user()->tenant;
        
        if (!$tenant) {
            return redirect()->route('tenant.dashboard')->with('error', 'Tenant account not found.');
        }
        
        $notifications = AdminNotification::forTenant($tenant->id)
            ->orderBy('sent_at', 'desc')
            ->paginate(20);

        $unreadCount = AdminNotification::forTenant($tenant->id)
            ->unread()
            ->count();

        return view('tenant.notifications.index', compact('notifications', 'unreadCount'));
    }

    public function show($id)
    {
        $notification = AdminNotification::findOrFail($id);
        
        // Check if this notification is for this tenant or for all
        $tenant = Auth::user()->tenant;
        if (!$tenant) {
            return redirect()->route('tenant.dashboard')->with('error', 'Tenant account not found.');
        }
        
        if ($notification->target_type === 'tenant' && $notification->target_id !== $tenant->id) {
            abort(403, 'Unauthorized access to this notification.');
        }

        // Mark as read if unread
        if (!$notification->is_read) {
            $notification->markAsRead();
        }

        return view('tenant.notifications.show', compact('notification'));
    }

    public function markAsRead($id)
    {
        $notification = AdminNotification::findOrFail($id);
        
        $tenant = Auth::user()->tenant;
        if (!$tenant) {
            return response()->json(['success' => false, 'message' => 'Tenant account not found.']);
        }
        
        if ($notification->target_type === 'tenant' && $notification->target_id !== $tenant->id) {
            return response()->json(['success' => false, 'message' => 'Unauthorized']);
        }

        $notification->markAsRead();
        return response()->json(['success' => true]);
    }

    public function markAllAsRead()
    {
        $tenant = Auth::user()->tenant;
        if (!$tenant) {
            return redirect()->route('tenant.dashboard')->with('error', 'Tenant account not found.');
        }
        
        AdminNotification::forTenant($tenant->id)
            ->unread()
            ->update([
                'is_read' => true,
                'read_at' => now()
            ]);

        return redirect()->route('tenant.notifications.index')
            ->with('success', 'All notifications marked as read.');
    }

    public function getUnreadCount()
    {
        try {
            $tenant = Auth::user()->tenant;
            if (!$tenant) {
                return response()->json(['count' => 0]);
            }
            
            $count = AdminNotification::forTenant($tenant->id)
                ->unread()
                ->count();

            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            // Log the error but return 0 to prevent frontend errors
            \Log::error('Error getting unread count: ' . $e->getMessage());
            return response()->json(['count' => 0]);
        }
    }

    public function getRecentNotifications()
    {
        $tenant = Auth::user()->tenant;
        if (!$tenant) {
            return response()->json([]);
        }
        
        $notifications = AdminNotification::forTenant($tenant->id)
            ->with(['sender'])
            ->unread()
            ->orderBy('sent_at', 'desc')
            ->limit(5)
            ->get();

        return response()->json($notifications);
    }
}
