<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\Manufacturer;
use Illuminate\Support\Facades\Storage;

class ManufacturerController extends Controller
{
    public function index()
    {
        $manufacturers = Manufacturer::orderBy('name')->paginate(10);
        return view('tenant.catalog.manufacturers.index', compact('manufacturers'));
    }

    public function create()
    {
        return view('tenant.catalog.manufacturers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'address' => 'nullable|string',
            'contact_no' => 'nullable|string|max:50',
            'contact_person' => 'nullable|string|max:255',
            'opening_balance' => 'nullable|numeric',
        ]);

        $data = $request->except('image'); // Use all validated data except image initially

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('manufacturers', 'public');
            $data['image'] = $path;
        }

        Manufacturer::create($data);
        return redirect()->route('tenant.manufacturers.index')->with('success', 'Manufacturer created successfully.');
    }

    public function edit($id)
    {
        $manufacturer = Manufacturer::on('tenant')->findOrFail($id);
        return view('tenant.catalog.manufacturers.edit', compact('manufacturer'));
    }

    public function update(Request $request, $id)
    {
        $manufacturer = Manufacturer::on('tenant')->findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'address' => 'nullable|string',
            'contact_no' => 'nullable|string|max:50',
            'contact_person' => 'nullable|string|max:255',
            'opening_balance' => 'nullable|numeric',
        ]);

        $data = $request->except('image');

        if ($request->hasFile('image')) {
            // Delete old image
            if ($manufacturer->image) {
                Storage::disk('public')->delete($manufacturer->image);
            }
            $path = $request->file('image')->store('manufacturers', 'public');
            $data['image'] = $path;
        }

        $manufacturer->update($data);
        return redirect()->route('tenant.manufacturers.index')->with('success', 'Manufacturer updated successfully.');
    }

    public function destroy($id)
    {
        $manufacturer = Manufacturer::on('tenant')->findOrFail($id);
        
        if ($manufacturer->image) {
            Storage::disk('public')->delete($manufacturer->image);
        }
        $manufacturer->delete();
        return redirect()->route('tenant.manufacturers.index')->with('success', 'Manufacturer deleted successfully.');
    }

    public function summary($id)
    {
        $manufacturer = Manufacturer::on('tenant')->findOrFail($id);
        // For now, this just shows the details. 
        // In the future, this can pull purchase history, current balance calculations, etc.
        return view('tenant.catalog.manufacturers.show', compact('manufacturer'));
    }
}
