<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\Customer;
use App\Models\Tenant\Setting;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class CustomerController extends Controller
{
    public function index()
    {
        $customers = Customer::orderBy('created_at', 'desc')->paginate(10);
        return view('tenant.customers.index', compact('customers'));
    }

    public function create()
    {
        return view('tenant.customers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'contact' => 'nullable|string|max:20',
            'phone' => 'nullable|string|max:20', // Accept phone if sent as phone
            'address' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'allow_credit' => 'boolean',
            'is_sub_dealer' => 'boolean',
        ]);

        if (isset($validated['phone'])) {
            $validated['contact'] = $validated['phone'];
            unset($validated['phone']);
        }

        $validated['is_active'] = $request->has('is_active') || $request->is_active === true;
        $validated['allow_credit'] = $request->has('allow_credit') || $request->allow_credit === true;
        $validated['is_sub_dealer'] = $request->has('is_sub_dealer') || $request->is_sub_dealer === true;

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('customers', 'public');
        }

        $customer = Customer::create($validated);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'customer' => $customer,
                'message' => 'Customer created successfully.'
            ]);
        }

        return redirect()->route('tenant.customers.index')->with('success', 'Customer created successfully.');
    }

    public function convertPoints(Request $request, $id)
    {
        $customer = Customer::on('tenant')->findOrFail($id);
        
        $request->validate([
            'points' => 'required|integer|min:1',
        ]);
        
        $pointsToConvert = $request->points;

        if ($customer->loyalty_points < $pointsToConvert) {
             return back()->with('error', 'Insufficient loyalty points.');
        }

        $conversionRate = Setting::get('loyalty_conversion_rate', '100'); // Points per 1 Currency Unit
        // Avoid division by zero
        if ($conversionRate <= 0) $conversionRate = 100;
        
        $amount = ($pointsToConvert / $conversionRate);

        DB::transaction(function() use ($customer, $pointsToConvert, $amount) {
            $customer->decrement('loyalty_points', $pointsToConvert);
            $customer->increment('wallet_balance', $amount);
        });

        return back()->with('success', "Converted $pointsToConvert points to $amount currency.");
    }

    public function show($id)
    {
        $customer = Customer::on('tenant')->findOrFail($id);
        $customer->load(['sales' => function($q) {
            $q->latest()->limit(10); // Last 10 transactions
        }, 'sales.items']); // Eager load sales items to avoid N+1 problem when summing quantity
        
        return view('tenant.customers.show', compact('customer'));
    }

    public function edit($id)
    {
        $customer = Customer::on('tenant')->findOrFail($id);
        return view('tenant.customers.edit', compact('customer'));
    }

    public function update(Request $request, $id)
    {
        $customer = Customer::on('tenant')->findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'contact' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'allow_credit' => 'boolean',
            'is_sub_dealer' => 'boolean',
        ]);

        $validated['is_active'] = $request->has('is_active');
        $validated['allow_credit'] = $request->has('allow_credit');
        $validated['is_sub_dealer'] = $request->has('is_sub_dealer');

        if ($request->hasFile('image')) {
            if ($customer->image) {
                Storage::disk('public')->delete($customer->image);
            }
            $validated['image'] = $request->file('image')->store('customers', 'public');
        }

        $customer->update($validated);

        return redirect()->route('tenant.customers.index')->with('success', 'Customer updated successfully.');
    }

    public function destroy($id)
    {
        $customer = Customer::on('tenant')->findOrFail($id);
        
        if ($customer->image) {
            Storage::disk('public')->delete($customer->image);
        }
        $customer->delete();
        return redirect()->route('tenant.customers.index')->with('success', 'Customer deleted successfully.');
    }
}
