<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\ChatSession;
use App\Models\ChatMessage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ChatController extends Controller
{
    public function index()
    {
        $tenant = Auth::user()->tenant;
        
        $session = ChatSession::where('tenant_id', $tenant->id)
            ->active()
            ->with(['messages' => function($query) {
                $query->orderBy('created_at', 'asc');
            }])
            ->first();

        // If no session exists, create one
        if (!$session) {
            $session = ChatSession::create([
                'tenant_id' => $tenant->id,
                'status' => 'active'
            ]);
        }

        // Mark admin messages as read
        if ($session->messages) {
            $session->messages()->where('sender_type', 'admin')
                ->where('is_read', false)
                ->update([
                    'is_read' => true,
                    'read_at' => now()
                ]);
        }

        return view('tenant.chat.index', compact('session'));
    }

    public function sendMessage(Request $request)
    {
        $request->validate([
            'message' => 'required|string|max:1000'
        ]);

        $tenant = Auth::user()->tenant;
        
        if (!$tenant) {
            return response()->json([
                'success' => false,
                'message' => 'Tenant account not found.'
            ], 404);
        }

        // Get or create chat session
        $session = ChatSession::where('tenant_id', $tenant->id)
            ->active()
            ->first();

        if (!$session) {
            $session = ChatSession::create([
                'tenant_id' => $tenant->id,
                'status' => 'active'
            ]);
        }

        $message = ChatMessage::create([
            'sender_id' => $tenant->id,
            'sender_type' => 'tenant',
            'receiver_id' => 1, // Admin user ID (you might want to make this configurable)
            'receiver_type' => 'admin',
            'message' => $request->message,
            'chat_session_id' => $session->id
        ]);

        // Update session last message time
        $session->update(['last_message_at' => now()]);

        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }

    public function getMessages()
    {
        $tenant = Auth::user()->tenant;
        
        $session = ChatSession::where('tenant_id', $tenant->id)
            ->active()
            ->first();

        if (!$session) {
            return response()->json([]);
        }

        $messages = ChatMessage::where('chat_session_id', $session->id)
            ->with(['sender'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Serialize messages with proper data structure
        $serializedMessages = $messages->map(function($message) {
            return [
                'id' => $message->id,
                'message' => $message->message,
                'sender_type' => $message->sender_type,
                'sender_id' => $message->sender_id,
                'sender_name' => $message->sender ? $message->sender->name : 'Unknown',
                'is_read' => $message->is_read,
                'read_at' => $message->read_at ? $message->read_at->format('Y-m-d H:i:s') : null,
                'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                'formatted_time' => $message->created_at->format('h:i A'),
                'formatted_date' => $message->created_at->format('M j, Y'),
                'timestamp' => $message->created_at->timestamp,
                'timezone' => config('app.timezone')
            ];
        });

        return response()->json($serializedMessages);
    }

    public function getUnreadCount()
    {
        $tenant = Auth::user()->tenant;
        
        $count = ChatMessage::where('receiver_id', $tenant->id)
            ->where('receiver_type', 'tenant')
            ->where('sender_type', 'admin')
            ->where('is_read', false)
            ->count();

        return response()->json(['count' => $count]);
    }

    public function getRecentNotifications()
    {
        $tenant = Auth::user()->tenant;
        if (!$tenant) {
            return response()->json([]);
        }
        
        $notifications = AdminNotification::forTenant($tenant->id)
            ->with(['sender'])
            ->unread()
            ->orderBy('sent_at', 'desc')
            ->limit(5)
            ->get();

        return response()->json($notifications);
    }

    public function streamMessages()
    {
        // For better compatibility, we'll use polling instead of SSE
        // This method now returns the latest messages for polling
        $tenant = Auth::user()->tenant;
        if (!$tenant) {
            return response()->json(['error' => 'Tenant not found'], 404);
        }

        $session = ChatSession::where('tenant_id', $tenant->id)
            ->active()
            ->first();

        if (!$session) {
            return response()->json(['error' => 'No active chat session'], 404);
        }

        $lastMessageId = request()->input('last_id', 0);
        
        // Get messages newer than last_id
        $messages = ChatMessage::where('chat_session_id', $session->id)
            ->where('id', '>', $lastMessageId)
            ->with(['sender'])
            ->orderBy('created_at', 'asc')
            ->get();

        $serializedMessages = $messages->map(function($message) {
            return [
                'id' => $message->id,
                'message' => $message->message,
                'sender_type' => $message->sender_type,
                'sender_id' => $message->sender_id,
                'sender_name' => $message->sender ? $message->sender->name : 'Unknown',
                'is_read' => $message->is_read,
                'read_at' => $message->read_at ? $message->read_at->format('Y-m-d H:i:s') : null,
                'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                'formatted_time' => $message->created_at->format('h:i A'),
                'formatted_date' => $message->created_at->format('M j, Y'),
                'timestamp' => $message->created_at->timestamp,
                'timezone' => config('app.timezone')
            ];
        });

        return response()->json([
            'messages' => $serializedMessages,
            'last_id' => $messages->max('id') ?? $lastMessageId,
            'timestamp' => now()->timestamp
        ]);
    }

    public function sendTypingIndicator(Request $request)
    {
        // This would typically store typing status in a separate table
        // For now, we'll just return success
        return response()->json(['success' => true]);
    }

    public function markAsRead($messageId)
    {
        $message = ChatMessage::findOrFail($messageId);
        
        // Only allow tenant to mark admin messages as read
        if ($message->sender_type === 'admin' && $message->receiver_type === 'tenant') {
            $message->markAsRead();
        }

        return response()->json(['success' => true]);
    }
}
