<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Tenant\Brand;
use Illuminate\Support\Facades\Storage;

class BrandController extends Controller
{
    public function index()
    {
        $brands = Brand::orderBy('name')->paginate(10);
        return view('tenant.catalog.brands.index', compact('brands'));
    }

    public function create()
    {
        return view('tenant.catalog.brands.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->only('name');

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('brands', 'public');
            $data['image'] = $path;
        }

        Brand::create($data);
        return redirect()->route('tenant.brands.index')->with('success', 'Brand created successfully.');
    }

    public function edit($id)
    {
        $brand = Brand::on('tenant')->findOrFail($id);
        return view('tenant.catalog.brands.edit', compact('brand'));
    }

    public function update(Request $request, $id)
    {
        $brand = Brand::on('tenant')->findOrFail($id);
        
        $request->validate([
            'name' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->only('name');

        if ($request->hasFile('image')) {
            if ($brand->image) {
                Storage::disk('public')->delete($brand->image);
            }
            $path = $request->file('image')->store('brands', 'public');
            $data['image'] = $path;
        }

        $brand->update($data);
        return redirect()->route('tenant.brands.index')->with('success', 'Brand updated successfully.');
    }

    public function destroy($id)
    {
        $brand = Brand::on('tenant')->findOrFail($id);
        
        if ($brand->image) {
            Storage::disk('public')->delete($brand->image);
        }
        $brand->delete();
        return redirect()->route('tenant.brands.index')->with('success', 'Brand deleted successfully.');
    }
}
