<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\LoginLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Show the application's login form for tenants.
     *
     * @return \Illuminate\View\View
     */
    public function showLoginForm()
    {
        if (Auth::check()) {
            return redirect()->route('tenant.dashboard');
        }
        return view('tenant.auth.login');
    }

    /**
     * Handle a login request to the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function login(Request $request)
    {
        // Identify if input is email or phone number
        $loginField = filter_var($request->login, FILTER_VALIDATE_EMAIL) ? 'email' : 'phone_number';

        $request->merge([$loginField => $request->login]);

        $request->validate([
            'login' => ['required', 'string'],
            'password' => ['required'],
        ]);

        $credentials = [
            $loginField => $request->login,
            'password' => $request->password,
        ];

        if (Auth::attempt($credentials, $request->boolean('remember'))) {
            $request->session()->regenerate();

            // Security Check: Only allow Tenant Users
            // Deny SaaS Admin (Role: Admin) from logging in here
            if (Auth::user()->hasRole('Admin')) {
                Auth::logout();
                throw ValidationException::withMessages([
                    'email' => 'This login form is for Tenant Users only. Admins please use the main login.',
                ]);
            }
            
            // Check if user belongs to a tenant
            if (!Auth::user()->tenant_id) {
                Auth::logout();
                 throw ValidationException::withMessages([
                    'email' => 'User does not belong to any tenant.',
                ]);
            }

            // Log Successful Login
            LoginLog::create([
                'email' => Auth::user()->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'status' => 'success',
            ]);

            return redirect()->intended(route('tenant.dashboard'));
        }

        // Log Failed Login
        LoginLog::create([
            'email' => $request->login,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'status' => 'failed',
        ]);

        throw ValidationException::withMessages([
            'login' => trans('auth.failed'),
        ]);
    }

    /**
     * Log the user out of the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect()->route('tenant.login');
    }
}
