<?php

namespace App\Http\Controllers\Developer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use App\Models\Tenant;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Mail;
use App\Mail\UserCredentialsMail;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with(['tenant', 'roles'])->orderBy('created_at', 'desc')->paginate(10);
        return view('developer.users.index', compact('users'));
    }

    public function create()
    {
        $tenants = Tenant::all();
        $roles = Role::all();
        return view('developer.users.create', compact('tenants', 'roles'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:8',
            'role' => 'required|string|exists:roles,name',
            'tenant_id' => 'nullable|exists:tenants,id',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'tenant_id' => $validated['tenant_id'],
            'status' => 'active',
        ]);

        $user->assignRole($validated['role']);

        try {
            Mail::to($user->email)->send(new UserCredentialsMail($user, $validated['password'], false));
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send credential email to new user: ' . $e->getMessage());
        }

        return redirect()->route('developer.users.index')->with('success', 'User created successfully.');
    }

    public function edit(User $user)
    {
        $tenants = Tenant::all();
        $roles = Role::all();
        
        $lastLogin = \App\Models\LoginLog::where('email', $user->email)
            ->where('status', 'success')
            ->latest()
            ->first();
            
        $lastIp = $lastLogin ? $lastLogin->ip_address : null;
        
        return view('developer.users.edit', compact('user', 'tenants', 'roles', 'lastIp'));
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'role' => 'required|string|exists:roles,name',
            'tenant_id' => 'nullable|exists:tenants,id',
            'status' => 'required|in:active,inactive',
        ]);

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'tenant_id' => $validated['tenant_id'] ?? null,
            'status' => $validated['status'],
        ]);

        if ($request->filled('password')) {
            $user->update(['password' => Hash::make($request->password)]);
        }

        $user->syncRoles([$validated['role']]);

        return redirect()->route('developer.users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if ($user->id === Auth::id()) {
            return back()->with('error', 'You cannot delete yourself.');
        }
        
        $user->delete();
        return redirect()->route('developer.users.index')->with('success', 'User deleted successfully.');
    }
}
