<?php

namespace App\Http\Controllers\Developer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Tenant;

class RequestMonitorController extends Controller
{
    public function index(Request $request)
    {
        $view = $request->query('view', 'all');
        $tenantId = $request->query('tenant_id');
        
        $query = DB::connection('mysql')->table('request_logs')
            ->leftJoin('tenants', 'request_logs.tenant_id', '=', 'tenants.id')
            ->leftJoin('users', 'request_logs.user_id', '=', 'users.id')
            ->select([
                'request_logs.*',
                'tenants.name as tenant_name',
                'users.name as user_name',
                'users.email as user_email'
            ]);

        // Filter by view
        if ($view === 'tenant' && $tenantId) {
            $query->where('request_logs.tenant_id', $tenantId);
        } elseif ($view === 'server') {
            $query->whereNull('request_logs.tenant_id');
        }

        $logs = $query->orderBy('request_logs.created_at', 'desc')
            ->paginate(50)
            ->withQueryString();

        $tenants = Tenant::all();

        // Statistics
        $stats = [
            'total_requests' => DB::connection('mysql')->table('request_logs')->count(),
            'avg_duration' => round(DB::connection('mysql')->table('request_logs')->avg('duration_ms') ?? 0, 2),
            'error_rate' => $this->calculateErrorRate(),
            'top_endpoints' => DB::connection('mysql')->table('request_logs')
                ->select('url', DB::raw('count(*) as count'))
                ->groupBy('url')
                ->orderBy('count', 'desc')
                ->limit(5)
                ->get(),
        ];

        return view('developer.requests.index', compact('logs', 'view', 'tenants', 'stats', 'tenantId'));
    }

    private function calculateErrorRate()
    {
        $total = DB::connection('mysql')->table('request_logs')->count();
        if ($total === 0) return 0;
        
        $errors = DB::connection('mysql')->table('request_logs')
            ->where('status_code', '>=', 400)
            ->count();
            
        return round(($errors / $total) * 100, 2);
    }

    public function getData(Request $request)
    {
        $view = $request->query('view', 'all');
        $tenantId = $request->query('tenant_id');
        
        $query = DB::connection('mysql')->table('request_logs')
            ->leftJoin('tenants', 'request_logs.tenant_id', '=', 'tenants.id')
            ->select([
                'request_logs.*',
                'tenants.name as tenant_name'
            ]);

        if ($view === 'tenant' && $tenantId) {
            $query->where('request_logs.tenant_id', $tenantId);
        } elseif ($view === 'server') {
            $query->whereNull('request_logs.tenant_id');
        }

        $logs = $query->orderBy('request_logs.created_at', 'desc')
            ->limit(20)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $logs
        ]);
    }

    public function clear()
    {
        DB::connection('mysql')->table('request_logs')->truncate();
        return redirect()->back()->with('success', 'Request logs cleared successfully.');
    }
}
