<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use App\Models\Tenant;
use App\Models\Invoice;
use App\Models\User;
use App\Models\Tenant\Sale;
use App\Models\Tenant\Product;
use App\Models\Tenant\Customer;

class DashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user) {
            return redirect('/login');
        }

        if ($user->hasRole('Admin')) {
            return $this->adminDashboard();
        } elseif ($user->hasRole('Manager')) {
            return $this->managerDashboard($user);
        } else {
            return view('dashboards.default');
        }
    }

    protected function adminDashboard()
    {
        $stats = [
            'total_tenants' => Tenant::count(),
            'active_tenants' => Tenant::where('status', 'active')->count(),
            'total_revenue' => Invoice::where('status', 'paid')->sum('amount'), // SaaS Revenue
            'total_users' => User::count(),
        ];

        $recentTenants = Tenant::orderBy('created_at', 'desc')->take(5)->get();
        $recentInvoices = Invoice::with('tenant')->orderBy('created_at', 'desc')->take(5)->get();

        return view('dashboards.admin', compact('stats', 'recentTenants', 'recentInvoices'));
    }

    protected function managerDashboard($user)
    {
        if ($user->status !== 'active') {
            return view('dashboards.inactive');
        }

        $tenant = $user->tenant;
        
        if (!$tenant) {
            abort(403, 'User does not belong to a tenant');
        }

        // Switch to Tenant Connection
        Config::set('database.connections.tenant.database', $tenant->database_name);
        DB::purge('tenant');
        
        // Fetch Stats (Using tenant connection implicitly via models or explicit connection if configured)
        // Since my Tenant models don't have explicit connection set (except the ones I just made?), 
        // I need to ensure they use 'tenant' connection or I set default.
        // For safety in this hybrid app, let's set default connection for this request scope.
        $originalConnection = DB::getDefaultConnection();
        DB::setDefaultConnection('tenant');

        try {
            $stats = [
                'revenue' => Sale::where('status', 'completed')->sum('total'),
                'orders' => Sale::count(),
                'products' => Product::count(),
                'customers' => Customer::count(),
            ];

            $recentSales = Sale::with(['customer', 'items'])
                ->orderBy('created_at', 'desc')
                ->take(5)
                ->get();

        } finally {
            // Restore connection just in case, though view rendering usually happens after
            // DB::setDefaultConnection($originalConnection); 
            // Actually, for the view to lazy load relationships, we must stay on tenant connection 
            // OR eagerly load everything. We eagerly loaded.
        }

        return view('dashboards.manager', compact('stats', 'recentSales', 'tenant'));
    }
}
