<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Tenant;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Mail;
use App\Mail\UserCredentialsMail;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with(['tenant', 'roles'])->orderBy('created_at', 'desc')->paginate(10);
        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        $tenants = Tenant::all();
        $roles = Role::all();
        return view('admin.users.create', compact('tenants', 'roles'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:8',
            'role' => 'required|string|exists:roles,name',
            'tenant_id' => 'nullable|exists:tenants,id',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'tenant_id' => $validated['tenant_id'],
            'status' => 'active',
        ]);

        $user->assignRole($validated['role']);

        try {
            Mail::to($user->email)->send(new UserCredentialsMail($user, $validated['password'], false));
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send credential email to new user: ' . $e->getMessage());
        }

        return redirect()->route('admin.users.index')->with('success', 'User created successfully.');
    }

    public function edit(User $user)
    {
        $tenants = Tenant::all();
        $roles = Role::all();
        
        // Get last login IP from logs
        $lastLogin = \App\Models\LoginLog::where('email', $user->email)
            ->where('status', 'success')
            ->latest()
            ->first();
            
        $lastIp = $lastLogin ? $lastLogin->ip_address : null;
        
        return view('admin.users.edit', compact('user', 'tenants', 'roles', 'lastIp'));
    }

    public function update(Request $request, User $user)
    {
        // Debug logging
        \Log::info('UserController@update called', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_name' => $user->name,
            'request_data_all' => $request->all(),
            'request_data_status' => $request->input('status'),
            'request_data_status_debug' => $request->input('status_debug'),
            'method' => $request->method(),
            'ajax' => $request->ajax(),
            'current_status_before' => $user->status,
            'submitted_status' => $request->input('status'),
        ]);

        // Check if status was submitted
        if (!$request->has('status')) {
            \Log::error('Status field not submitted in request', ['user_id' => $user->id]);
            return back()->with('error', 'Status field is required')->withInput();
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'role' => 'required|string|exists:roles,name',
            'tenant_id' => 'nullable|exists:tenants,id',
            'status' => 'required|in:active,inactive',
        ]);

        // Log validated data
        \Log::info('UserController@update validation passed', [
            'user_id' => $user->id,
            'validated_status' => $validated['status'],
            'current_status_before_update' => $user->status,
        ]);

        // Check if status is actually changing
        if ($user->status === $validated['status']) {
            \Log::warning('UserController@update status not changing', [
                'user_id' => $user->id,
                'current_status' => $user->status,
                'new_status' => $validated['status'],
                'status_same' => true
            ]);
        } else {
            \Log::info('UserController@update status changing', [
                'user_id' => $user->id,
                'from_status' => $user->status,
                'to_status' => $validated['status'],
            ]);
        }

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'tenant_id' => $validated['tenant_id'] ?? null,
            'status' => $validated['status'],
        ]);

        // Log after update - use fresh() to get latest data from database
        $freshUser = $user->fresh();
        \Log::info('UserController@update after database update', [
            'user_id' => $user->id,
            'old_status' => $user->status,
            'new_status_from_fresh' => $freshUser->status,
            'validated_status' => $validated['status'],
            'status_changed' => $freshUser->status !== $user->status,
            'status_matches_validated' => $freshUser->status === $validated['status'],
        ]);

        if ($request->filled('password')) {
            $user->update(['password' => Hash::make($request->password)]);
        }

        $user->syncRoles([$validated['role']]);

        \Log::info('UserController@update completed successfully', [
            'user_id' => $user->id,
            'final_status' => $user->fresh()->status,
        ]);

        return redirect()->route('admin.users.index')->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if ($user->id === Auth::id()) {
            return back()->with('error', 'You cannot delete yourself.');
        }
        
        $user->delete();
        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully.');
    }
}
