<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Plan;
use App\Models\TenantCategory;
use Illuminate\Support\Str;

class PlanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $plans = Plan::with('category')->get();
        return view('admin.plans.index', compact('plans'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = TenantCategory::where('is_active', true)->get();
        return view('admin.plans.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:basic,normal,vip,commercial',
            'price' => 'required|numeric|min:0',
            'interval' => 'required|in:daily,monthly,yearly',
            'description' => 'nullable|string',
            'features' => 'nullable|array',
            'tenant_category_id' => 'required|exists:tenant_categories,id',
        ]);

        $validated['slug'] = Str::slug($request->name . '-' . $request->interval);

        Plan::create($validated);

        return redirect()->route('admin.plans.index')->with('success', 'Plan created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Plan $plan)
    {
        return view('admin.plans.show', compact('plan'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Plan $plan)
    {
        $categories = TenantCategory::where('is_active', true)->get();
        return view('admin.plans.edit', compact('plan', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Plan $plan)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:basic,normal,vip,commercial',
            'price' => 'required|numeric|min:0',
            'interval' => 'required|in:daily,monthly,yearly',
            'description' => 'nullable|string',
            'features' => 'nullable|array',
            'tenant_category_id' => 'required|exists:tenant_categories,id',
        ]);

        $validated['slug'] = Str::slug($request->name . '-' . $request->interval);
        
        $plan->update($validated);

        return redirect()->route('admin.plans.index')->with('success', 'Plan updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Plan $plan)
    {
        $plan->delete();
        return redirect()->route('admin.plans.index')->with('success', 'Plan deleted successfully.');
    }
}
