<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminNotification;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    public function index()
    {
        $notifications = AdminNotification::with(['sender'])
            ->orderBy('sent_at', 'desc')
            ->paginate(20);

        $unreadCount = AdminNotification::unread()->count();

        return view('admin.notifications.index', compact('notifications', 'unreadCount'));
    }

    public function create()
    {
        $tenants = Tenant::orderBy('name')->get();
        return view('admin.notifications.create', compact('tenants'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:1000',
            'type' => 'required|in:info,warning,success,error',
            'target_type' => 'required|in:all,tenant,customer',
            'tenant_id' => 'required_if:target_type,tenant|exists:tenants,id'
        ]);

        $notification = AdminNotification::create([
            'title' => $request->title,
            'message' => $request->message,
            'type' => $request->type,
            'target_type' => $request->target_type,
            'target_id' => $request->target_type === 'tenant' ? $request->tenant_id : null,
            'sent_by' => Auth::id(),
            'sent_at' => now()
        ]);

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Notification sent successfully!');
    }

    public function show($id)
    {
        $notification = AdminNotification::with(['sender'])
            ->findOrFail($id);

        // Mark as read if unread
        if (!$notification->is_read) {
            $notification->markAsRead();
        }

        return view('admin.notifications.show', compact('notification'));
    }

    public function markAsRead($id)
    {
        $notification = AdminNotification::findOrFail($id);
        $notification->markAsRead();

        return response()->json(['success' => true]);
    }

    public function markAllAsRead()
    {
        AdminNotification::unread()->update([
            'is_read' => true,
            'read_at' => now()
        ]);

        return redirect()->route('admin.notifications.index')
            ->with('success', 'All notifications marked as read.');
    }

    public function destroy($id)
    {
        $notification = AdminNotification::findOrFail($id);
        $notification->delete();

        return redirect()->route('admin.notifications.index')
            ->with('success', 'Notification deleted successfully.');
    }

    public function getUnreadCount()
    {
        $count = AdminNotification::unread()->count();
        return response()->json(['count' => $count]);
    }

    public function getRecentNotifications()
    {
        $notifications = AdminNotification::with(['sender'])
            ->unread()
            ->orderBy('sent_at', 'desc')
            ->limit(5)
            ->get();

        return response()->json($notifications);
    }
}
