<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ChatSession;
use App\Models\ChatMessage;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ChatController extends Controller
{
    public function index()
    {
        $sessions = ChatSession::with(['tenant', 'latestMessage'])
            ->active()
            ->orderBy('last_message_at', 'desc')
            ->paginate(20);

        $unreadCount = ChatMessage::forAdmin()->unread()->count();

        return view('admin.chat.index', compact('sessions', 'unreadCount'));
    }

    public function show($id)
    {
        $session = ChatSession::with(['tenant', 'messages' => function($query) {
            $query->orderBy('created_at', 'asc');
        }])->findOrFail($id);

        // Mark messages as read
        ChatMessage::where('chat_session_id', $id)
            ->forAdmin()
            ->unread()
            ->update([
                'is_read' => true,
                'read_at' => now()
            ]);

        return view('admin.chat.show', compact('session'));
    }

    public function sendMessage(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string|max:1000'
        ]);

        $session = ChatSession::findOrFail($id);

        $message = ChatMessage::create([
            'sender_id' => Auth::id(),
            'sender_type' => 'admin',
            'receiver_id' => $session->tenant_id,
            'receiver_type' => 'tenant',
            'message' => $request->message,
            'chat_session_id' => $id
        ]);

        // Update session last message time
        $session->update(['last_message_at' => now()]);

        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }

    public function startChat(Request $request)
    {
        $request->validate([
            'tenant_id' => 'required|exists:tenants,id'
        ]);

        $tenant = Tenant::findOrFail($request->tenant_id);

        // Check if chat session already exists
        $existingSession = ChatSession::where('tenant_id', $tenant->id)
            ->active()
            ->first();

        if ($existingSession) {
            return redirect()->route('admin.chat.show', $existingSession->id);
        }

        // Create new chat session
        $session = ChatSession::create([
            'tenant_id' => $tenant->id,
            'admin_id' => Auth::id(),
            'status' => 'active'
        ]);

        return redirect()->route('admin.chat.show', $session->id);
    }

    public function closeSession($id)
    {
        $session = ChatSession::findOrFail($id);
        $session->update(['status' => 'closed']);

        return redirect()->route('admin.chat.index')
            ->with('success', 'Chat session closed successfully.');
    }

    public function getMessages($id)
    {
        $messages = ChatMessage::where('chat_session_id', $id)
            ->with(['sender'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Serialize messages with proper data structure
        $serializedMessages = $messages->map(function($message) {
            return [
                'id' => $message->id,
                'message' => $message->message,
                'sender_type' => $message->sender_type,
                'sender_id' => $message->sender_id,
                'sender_name' => $message->sender ? $message->sender->name : 'Unknown',
                'is_read' => $message->is_read,
                'read_at' => $message->read_at ? $message->read_at->format('Y-m-d H:i:s') : null,
                'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                'formatted_time' => $message->created_at->format('h:i A'),
                'formatted_date' => $message->created_at->format('M j, Y'),
                'timestamp' => $message->created_at->timestamp,
                'timezone' => config('app.timezone')
            ];
        });

        return response()->json($serializedMessages);
    }

    public function getUnreadCount()
    {
        $count = ChatMessage::forAdmin()->unread()->count();
        return response()->json(['count' => $count]);
    }

    public function searchTenants(Request $request)
    {
        $search = $request->input('q');
        
        $tenants = Tenant::where('name', 'like', "%{$search}%")
            ->orWhere('domain', 'like', "%{$search}%")
            ->limit(10)
            ->get();

        return response()->json($tenants);
    }

    public function streamMessages($id)
    {
        // For better compatibility, we'll use polling instead of SSE
        // This method now returns the latest messages for polling
        $session = ChatSession::findOrFail($id);

        // Check if admin has access to this session
        // Add your authorization logic here if needed

        $lastMessageId = request()->input('last_id', 0);
        
        // Get messages newer than last_id
        $messages = ChatMessage::where('chat_session_id', $session->id)
            ->where('id', '>', $lastMessageId)
            ->with(['sender'])
            ->orderBy('created_at', 'asc')
            ->get();

        $serializedMessages = $messages->map(function($message) {
            return [
                'id' => $message->id,
                'message' => $message->message,
                'sender_type' => $message->sender_type,
                'sender_id' => $message->sender_id,
                'sender_name' => $message->sender ? $message->sender->name : 'Unknown',
                'is_read' => $message->is_read,
                'read_at' => $message->read_at ? $message->read_at->format('Y-m-d H:i:s') : null,
                'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                'formatted_time' => $message->created_at->format('h:i A'),
                'formatted_date' => $message->created_at->format('M j, Y'),
                'timestamp' => $message->created_at->timestamp,
                'timezone' => config('app.timezone')
            ];
        });

        return response()->json([
            'messages' => $serializedMessages,
            'last_id' => $messages->max('id') ?? $lastMessageId,
            'timestamp' => now()->timestamp
        ]);
    }

    public function sendTypingIndicator($id, Request $request)
    {
        // This would typically store typing status in a separate table
        // For now, we'll just return success
        return response()->json(['success' => true]);
    }
}
