<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AppSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class AiController extends Controller
{
    public function generatePrompt(Request $request)
    {
        $request->validate([
            'context' => 'required|string',
        ]);

        $activeApis = $this->getActiveApis();

        if (empty($activeApis)) {
            // Fallback to .env if no database settings found
            $envKey = config('services.openai.key') ?: env('OPENAI_API_KEY');
            if ($envKey) {
                $activeApis[] = [
                    'provider' => 'openai',
                    'key' => $envKey,
                    'model' => 'gpt-4o-mini'
                ];
            }
        }

        if (empty($activeApis)) {
            return response()->json([
                'success' => false,
                'message' => 'No AI API configured. Please go to Admin Settings > AI Configuration.',
            ], 400);
        }

        // Randomize order to distribute load
        shuffle($activeApis);

        $errors = [];
        foreach ($activeApis as $api) {
            try {
                $result = $this->callApi($api, $request->context);
                if ($result['success']) {
                    return response()->json([
                        'success' => true,
                        'prompt' => $result['content'],
                        'provider' => $api['provider']
                    ]);
                }
                Log::warning("AI Provider Failed ({$api['provider']}): " . $result['message']);
                $errors[] = "{$api['provider']}: {$result['message']}";
            } catch (\Exception $e) {
                Log::error("AI Error ({$api['provider']}): " . $e->getMessage());
                $errors[] = "{$api['provider']}: " . $e->getMessage();
            }
        }

        return response()->json([
            'success' => false,
            'message' => 'All configured AI APIs failed. Errors: ' . implode(' | ', $errors),
        ], 500);
    }

    private function getActiveApis()
    {
        $apis = [];
        
        // OpenAI
        if (AppSetting::get('ai_openai_enable')) {
            $apis[] = [
                'provider' => 'openai',
                'key' => AppSetting::get('ai_openai_api_key'),
                'model' => AppSetting::get('ai_openai_model', 'gpt-4o-mini')
            ];
        }

        // Anthropic
        if (AppSetting::get('ai_anthropic_enable')) {
            $apis[] = [
                'provider' => 'anthropic',
                'key' => AppSetting::get('ai_anthropic_api_key'),
                'model' => AppSetting::get('ai_anthropic_model', 'claude-3-haiku-20240307')
            ];
        }

        // Gemini
        if (AppSetting::get('ai_gemini_enable')) {
            $apis[] = [
                'provider' => 'gemini',
                'key' => AppSetting::get('ai_gemini_api_key'),
                'model' => AppSetting::get('ai_gemini_model', 'gemini-1.5-flash')
            ];
        }

        // OpenRouter
        if (AppSetting::get('ai_openrouter_enable')) {
            $key = AppSetting::get('api_openrouter_key');
            Log::info('OpenRouter Key Check: ' . ($key ? 'Key Found (Starts with: ' . substr($key, 0, 5) . '...)' : 'Key NOT Found'));
            $apis[] = [
                'provider' => 'openrouter',
                'key' => $key,
                'model' => AppSetting::get('ai_openrouter_model', 'google/gemini-2.0-flash-exp:free')
            ];
        }

        return array_filter($apis, fn($api) => !empty($api['key']));
    }

    private function callApi($api, $context)
    {
        switch ($api['provider']) {
            case 'openai':
                return $this->callOpenAI($api, $context);
            case 'anthropic':
                return $this->callAnthropic($api, $context);
            case 'gemini':
                return $this->callGemini($api, $context);
            case 'openrouter':
                return $this->callOpenRouter($api, $context);
            default:
                return ['success' => false, 'message' => 'Unknown provider'];
        }
    }

    private function callOpenAI($api, $context)
    {
        $response = Http::withToken($api['key'])
            ->post('https://api.openai.com/v1/chat/completions', [
                'model' => $api['model'],
                'messages' => [
                    ['role' => 'system', 'content' => $this->getSystemPrompt()],
                    ['role' => 'user', 'content' => "Enhance and refine:\n\n" . $context],
                ],
                'temperature' => 0.7,
            ]);

        if ($response->successful()) {
            return [
                'success' => true,
                'content' => $response->json()['choices'][0]['message']['content']
            ];
        }

        $error = $response->json()['error']['message'] ?? 'OpenAI API Error';
        return ['success' => false, 'message' => $error];
    }

    private function callAnthropic($api, $context)
    {
        $response = Http::withHeaders([
            'x-api-key' => $api['key'],
            'anthropic-version' => '2023-06-01',
            'content-type' => 'application/json',
        ])->post('https://api.anthropic.com/v1/messages', [
            'model' => $api['model'],
            'max_tokens' => 4000,
            'system' => $this->getSystemPrompt(),
            'messages' => [
                ['role' => 'user', 'content' => "Enhance and refine the following technical configuration into a comprehensive developer prompt:\n\n" . $context],
            ],
        ]);

        if ($response->successful()) {
            return [
                'success' => true,
                'content' => $response->json()['content'][0]['text']
            ];
        }

        $error = $response->json()['error']['message'] ?? 'Anthropic API Error';
        return ['success' => false, 'message' => $error];
    }

    private function callGemini($api, $context)
    {
        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$api['model']}:generateContent?key=" . $api['key'];
        
        $response = Http::post($url, [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $this->getSystemPrompt() . "\n\nEnhance and refine:\n\n" . $context]
                    ]
                ]
            ],
            'generationConfig' => [
                'temperature' => 0.7,
                'maxOutputTokens' => 4000,
            ]
        ]);

        if ($response->successful()) {
            return [
                'success' => true,
                'content' => $response->json()['candidates'][0]['content']['parts'][0]['text']
            ];
        }

        $error = $response->json()['error']['message'] ?? 'Gemini API Error';
        return ['success' => false, 'message' => $error];
    }

    private function callOpenRouter($api, $context)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $api['key'],
                'Content-Type' => 'application/json',
                'HTTP-Referer' => config('app.url'),
                'X-Title' => config('app.name'),
            ])->timeout(30)->post('https://openrouter.ai/api/v1/chat/completions', [
                'model' => $api['model'],
                'messages' => [
                    ['role' => 'system', 'content' => $this->getSystemPrompt()],
                    ['role' => 'user', 'content' => "Enhance and refine:\n\n" . $context],
                ],
                'temperature' => 0.7,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                if (isset($data['choices'][0]['message']['content'])) {
                    return [
                        'success' => true,
                        'content' => $data['choices'][0]['message']['content']
                    ];
                }
                return ['success' => false, 'message' => 'Invalid response format from OpenRouter'];
            }

            $errorData = $response->json();
            $errorMessage = $errorData['error']['message'] ?? $response->body() ?: 'OpenRouter API Error';
            Log::error("OpenRouter Details: " . $response->body());
            
            return ['success' => false, 'message' => $errorMessage];
        } catch (\Exception $e) {
            Log::error("OpenRouter Exception: " . $e->getMessage());
            return ['success' => false, 'message' => 'Connection error: ' . $e->getMessage()];
        }
    }

    private function getSystemPrompt()
    {
        return 'You are an expert SaaS Architect. Your task is to take a raw technical configuration summary and turn it into a highly detailed, professional developer instruction prompt for building a multi-tenant Laravel application. Focus on database structure, specific controllers, and modular logic.';
    }
}
