<?php

namespace App\Exports;

use App\Models\Tenant\Product;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Illuminate\Http\Request;
use Carbon\Carbon;

class StockReportExport implements FromQuery, WithHeadings, WithMapping
{
    protected $categoryId;
    protected $stockStatus;
    protected $search;
    protected $productId;
    protected $startDate;
    protected $endDate;

    public function __construct($categoryId, $stockStatus, $search, $productId, $startDate, $endDate)
    {
        $this->categoryId = $categoryId;
        $this->stockStatus = $stockStatus;
        $this->search = $search;
        $this->productId = $productId;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
    }

    public function query()
    {
        $query = Product::with(['variants', 'category', 'unit'])
            ->select('products.*', 'categories.name as category_name', 'units.name as unit_name')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->join('units', 'products.unit_id', '=', 'units.id');

        if ($this->categoryId) {
            $query->where('products.category_id', $this->categoryId);
        }

        if ($this->startDate) {
            $query->whereDate('products.created_at', '>=', $this->startDate);
        }

        if ($this->endDate) {
            $query->whereDate('products.created_at', '<=', $this->endDate);
        }

        if ($this->search) {
            $query->where(function($q) {
                $q->where('products.name', 'like', "%{$this->search}%")
                  ->orWhereHas('variants', function($vq) {
                      $vq->where('name', 'like', "%{$this->search}%")
                         ->orWhere('sku', 'like', "%{$this->search}%")
                         ->orWhere('barcode', 'like', "%{$this->search}%");
                  });
            });
        }

        return $query;
    }

    public function headings(): array
    {
        return [
            'Product Name',
            'Variant Name',
            'SKU',
            'Barcode',
            'Category',
            'Unit',
            'Current Stock',
            'Purchase Price',
            'Selling Price',
            'Stock Value (Purchase)',
            'Stock Value (Selling)',
            'Status',
            'Created At',
        ];
    }

    public function map($product): array
    {
        $data = [];
        foreach ($product->variants as $variant) {
            $status = 'Normal';
            if ($variant->stock <= 0) {
                $status = 'Out of Stock';
            } elseif ($variant->stock < 10) { // Assuming 10 is the low stock threshold
                $status = 'Low Stock';
            }

            // Apply stock status filter if set
            if ($this->stockStatus !== 'all') {
                if ($this->stockStatus === 'low' && $status !== 'Low Stock') {
                    continue;
                }
                if ($this->stockStatus === 'out' && $status !== 'Out of Stock') {
                    continue;
                }
                if ($this->stockStatus === 'normal' && $status !== 'Normal') {
                    continue;
                }
            }

            // Apply product ID filter if set
            if ($this->productId && $product->id != $this->productId) {
                continue;
            }

            $data[] = [
                $product->name,
                $variant->name,
                $variant->sku,
                $variant->barcode,
                $product->category_name,
                $product->unit_name,
                $variant->stock,
                $variant->purchase_price,
                $variant->price,
                $variant->stock * $variant->purchase_price,
                $variant->stock * $variant->price,
                $status,
                Carbon::parse($product->created_at)->format('Y-m-d H:i:s'),
            ];
        }
        return $data;
    }
}
